/*
 * =====================================================================================
 *
 *       Filename:  GraphAlgorithm_BusackerGowen.hpp
 *
 *    Description:  Implementation of the algorithm of Busacker and Gowen
 *
 *        Version:  1.0
 *        Created:  12/14/2008 08:11:49 PM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef GraphAlgorithm_BusackerGowen_H
#define GraphAlgorithm_BusackerGowen_H

#include "GraphAlgorithm_Basic.hpp"
#include "GraphWriter.h"

/*
 * =====================================================================================
 *        Class:  GraphAlgorithm_BusackerGowen
 *  Description:  The algorithm of Busacker and Gowen works on a flow network
 *                with integer capacity. Every edge is associated with a cost,
 *                this algorithm will find a flow such that the sum of
 *                cost(e)*flow(e) is minimized and the flow from the source to
 *                the sink is equal to a specified value.
 *  
 * =====================================================================================
 */
template < typename COST_TYPE >
class GraphAlgorithm_BusackerGowen
{
	public:

		/* ====================  LIFECYCLE     ======================================= */
		GraphAlgorithm_BusackerGowen ();                             /* constructor */
		~GraphAlgorithm_BusackerGowen ();                             /* destructor */

		/*
		 * Prepare the buf of the algorithm. A flow network will be created in this
		 * function
		 */
		void
		PrepareBuf(Graph* p_graph, int source, int sink, int* capacity, COST_TYPE* cost);

		/*
		 * Calculate the optimal flow
		 */
		bool 
		OptimalFlow(int flow_value);

		/*
		 * Free all the buffers and the flow network
		 */
		void
		ClearBuf();

		/*
		 * When the calculation is done, find the final flow
		 */
		void
		GetFlow(int* flow);

	protected:

	private:
		COST_TYPE* mAuxCost;
		COST_TYPE* mCostBack;
		int* mAuxCap;
		bool* mbAux;
		int* mOriEdge;
		int* mRevEdge;
		int* mAuxFlow;
		int* mBtNode;
		COST_TYPE* mDistance;
		COST_TYPE* mDistanceBefore;
		int* mMaskBuf;
		int mSource;
		int mSink;
		int mCurrFlow;
	
		Graph* mpGraph;
		GraphEx<int> mAuxNet;


}; /* ----------  end of template class GraphAlgorithm_BusackerGowen  ---------- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_BusackerGowen
 *      Method:  GraphAlgorithm_BusackerGowen
 * Description:  constructor
 *--------------------------------------------------------------------------------------
 */

	template < class COST_TYPE >
GraphAlgorithm_BusackerGowen<COST_TYPE>::GraphAlgorithm_BusackerGowen()
{
	mAuxCost = 0;
	mCostBack = 0;
	mAuxCap = 0;
	mAuxFlow = 0;
	mbAux = 0;
	mOriEdge = 0;
	mRevEdge = 0;
	mBtNode = 0;
	mDistance = 0;
	mDistanceBefore = 0;
	mCurrFlow = 0;
	mMaskBuf = 0;
}		/* -----  end of method GraphAlgorithm_BusackerGowen<COST_TYPE>::GraphAlgorithm_BusackerGowen (constructor) ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_BusackerGowen
 *      Method:  ~GraphAlgorithm_BusackerGowen
 * Description:  destructor
 *--------------------------------------------------------------------------------------
 */

	template < class COST_TYPE >
GraphAlgorithm_BusackerGowen<COST_TYPE>::~GraphAlgorithm_BusackerGowen()
{
	ClearBuf();
}		/* -----  end of method GraphAlgorithm_BusackerGowen<COST_TYPE>::GraphAlgorithm_BusackerGowen (destructor) ----- */



/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_BusackerGowen
 *      Method:  PrepareBuf
 * Description:  Prepare buffers for the algorithm. The flow network will be created in
 *               this function.
 *       Param:  source    :    The internal ID of the source node
 *               sink      :    The internal ID of the sink node
 *               capacity  :    An array indexed by internal edge IDs of the graph 
 *               cost      :    An array indexed by internal edge IDs of the graph
 *      Return:  None
 *        NOTE:  Make sure that the graph has no reversal edges, i.e. only one
 *		         of the two edges of (n1, n2) and (n2, n1) could exist. Or else, the 
 *		         algorithm fails.
 *--------------------------------------------------------------------------------------
 */

	template < class COST_TYPE >
	void 
GraphAlgorithm_BusackerGowen<COST_TYPE>::PrepareBuf ( 
		Graph* p_graph, int source, int sink, int* capacity, COST_TYPE* cost)
{
	if (mAuxCost) ClearBuf();

	mAuxCost = new COST_TYPE[p_graph->EdgeCnt()*2];
	mCostBack = new COST_TYPE[p_graph->EdgeCnt()*2];
	mAuxCap = new int[p_graph->EdgeCnt()*2];
	mbAux = new bool[p_graph->EdgeCnt()*2];
	mOriEdge = new int[p_graph->EdgeCnt()*2];
	mRevEdge = new int[p_graph->EdgeCnt()*2];
	mAuxFlow = new int[p_graph->EdgeCnt()*2];
	mBtNode = new int[p_graph->NodeCnt()];
	mDistance = new COST_TYPE[p_graph->NodeCnt()];
	mDistanceBefore = new COST_TYPE[p_graph->NodeCnt()];
	mMaskBuf = new int[p_graph->EdgeCnt()*2];
	mCurrFlow = 0;

	for (int i = 0; i < p_graph->NodeCnt(); i++)
		mDistanceBefore[i] = 0;

	int mask_cnt = 0;
	// Create the auxiliary network
	mAuxNet.BeginAddNodeOrEdge();
	const int* edges = p_graph->Edges();
	for (int i = 0; i < p_graph->EdgeCnt(); i++)
	{
		int edge = edges[i];
		int ori_edge_id = mAuxNet.AddEdgeEx(p_graph->FromNode(edge), p_graph->ToNode(edge), edge);
		mCostBack[ori_edge_id] = cost[edge];
		mAuxCap[ori_edge_id] = capacity[edge];
		// Init the flow of the network with given flow
		mAuxFlow[ori_edge_id] = 0;
		mbAux[ori_edge_id] = false;

		// Because p_graph does not contain reversal edges, so this AddEdgeEx
		// should work
		int rev_edge_id = mAuxNet.AddEdgeEx(p_graph->ToNode(edge), p_graph->FromNode(edge), 0);
		if (-1 == rev_edge_id)
			cerr << "ERROR : The graph passed to GraphAlgorithm_BusackerGowen should not \
				contain any reversal edges" << endl;
		mCostBack[rev_edge_id] = -cost[edge];
		mAuxCap[rev_edge_id] = 0;
		mAuxFlow[rev_edge_id] = 0;
		mbAux[rev_edge_id] = true;

		mOriEdge[rev_edge_id] = ori_edge_id;
		mOriEdge[ori_edge_id] = -1;
		mRevEdge[ori_edge_id] = rev_edge_id;
		mRevEdge[rev_edge_id] = -1;

		mMaskBuf[mask_cnt++] = rev_edge_id;
	}

	mAuxNet.EndAddNodeOrEdge();

	memcpy(mAuxCost, mCostBack, sizeof(COST_TYPE)*mAuxNet.EdgeCnt());

	mSource = mAuxNet.GetNodeInID(source);
	mSink = mAuxNet.GetNodeInID(sink);

	mAuxNet.MaskAll(false);
	mAuxNet.MaskEdge(mMaskBuf, mask_cnt);
	
#ifdef DEBUG1
	cout << "DEBUG1 : Source is " << mSource << endl;
	cout << "DEBUG1 : Sink is " << mSink << endl;
	GraphWriter writer;
	mAuxNet.Write(&writer);
#endif

	return ;
}		/* -----  end of method GraphAlgorithm_BusackerGowen<COST_TYPE>::PrepareBuf  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_BusackerGowen
 *      Method:  ClearBuf
 * Description:  Clear the buffers
 *       Param:  
 *      Return:  None
 *--------------------------------------------------------------------------------------
 */

	template < class COST_TYPE >
	void 
GraphAlgorithm_BusackerGowen<COST_TYPE>::ClearBuf()
{
	if (mAuxCost)
	{
		delete[] mAuxCost;
		mAuxCost = 0;
		delete[] mCostBack;
		mCostBack = 0;
		delete[] mAuxCap;
		mAuxCap = 0;
		delete[] mAuxFlow;
		mAuxFlow = 0;
		delete[] mbAux;
		mbAux = 0;
		delete[] mOriEdge;
		mOriEdge = 0;
		delete[] mRevEdge;
		mRevEdge = 0;
		delete[] mBtNode;
		mBtNode = 0;
		delete[] mDistance;
		mDistance = 0;
		delete[] mDistanceBefore;
		mDistanceBefore= 0;
		delete[] mMaskBuf;
		mMaskBuf = 0;
		
		mAuxNet.Destroy();
	}
	return ;
}		/* -----  end of method GraphAlgorithm_BusackerGowen<COST_TYPE>::ClearBuf ----- */



/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_BusackerGowen
 *      Method:  OptimalFlow
 * Description:  Refers to the description of the class
 *       Param:  flow_value:    The specified flow value
 *      Return:  true  :   If there is a flow has the specified value
 *               false :   or else.
 *        Note:  This function can be called multiple times after the PrepareBuf is called
 *               with increasing 'flow_value'. If a smaller flow_value than last round, 
 *               this function will return true directly.
 *--------------------------------------------------------------------------------------
 */
	template < class COST_TYPE >
	bool 
GraphAlgorithm_BusackerGowen<COST_TYPE>::OptimalFlow(int flow_value)
{
	bool b_exist = true;
	while (b_exist && mCurrFlow < flow_value)
	{
		/*
		cout << "DEBUG : Current flow value " << mCurrFlow << " flow_value = " << flow_value << endl;
		*/

		// TRICK : Because Dijkstra algorithm does not allow negative costs on the edges,
		// the costs are adjusted by the trick due to Edmonds and Karp (1972) "Theoretical
		// improvements in algorithmic efficiency for network flow problems". See also
		// p275 "Graphs, Networks and Algorithms" by Dieter Jungnickel.
		// The shortest path for a node from the source under the adjusted cost is the same
		// as that under the unadjusted cost.
		// At the beginning, the cost needs not to be adjusted, because all the cost are
		// nonnegative.

		if (!GraphAlgorithm_Basic::ShortestDistance_Dijkstra<COST_TYPE>(&mAuxNet, mSource, mSink, mAuxCost, mDistance, mBtNode))
			b_exist = false;
		else
		{
			// cout << "DEBUG : Find the smallest capacity in the augment path" << endl;
			// Find the smallest capacity in the augment path
			int s_cap = numeric_limits<int>::max();
			int next = mSink;
			while (next != mSource)
			{
#ifdef DEBUG1
				cout << "DEBUG1 : next = " << next << endl;
#endif
				int edge = mAuxNet.EdgeID(mBtNode[next], next);
				if (edge == -1)
				{
					cerr << "ERROR : This edge " << mBtNode[next] << "," << next << "should not be -1 in " << __func__<< endl;
				}
				if (s_cap > mAuxCap[edge])
					s_cap = mAuxCap[edge];
				next = mBtNode[next];
			}

#ifdef DEBUG1
			cout << "DEBUG1 : Augment path capacity = " << s_cap << endl;
#endif

			if (s_cap > flow_value - mCurrFlow)
				s_cap = flow_value - mCurrFlow;
			mCurrFlow += s_cap;
			
			// cout << "DEBUG : Augment the flow" << endl;
			// Augment the flow
			next = mSink;
			while (next != mSource)
			{
				int edge = mAuxNet.EdgeID(mBtNode[next], next);
				if (mbAux[edge])
				{
					mAuxCap[edge] -= s_cap;
					mAuxFlow[mOriEdge[edge]] -= s_cap;
				}
				else
				{
					mAuxCap[mRevEdge[edge]] += s_cap;
					mAuxFlow[edge] += s_cap;
				}
				next = mBtNode[next];
			}

			/*
			cout << "DEBUG : distance before" << endl;
			const int* nodes = mAuxNet.Nodes();
			for (int i = 0; i < mAuxNet.NodeCnt(); i++)
			{
				int node = nodes[i];
				cout << mDistance[node] << "\t";
			}
			cout << endl;
			cout << "DEBUG : TRICK : Adjust the distance back to the real distance under the cost before adjustment." << endl;
			*/
			// TRICK : Adjust the distance back to the real distance under the cost before adjustment.
			// This step is necessary to calculate the next round of adjusted cost.
			const int* nodes = mAuxNet.Nodes();
			for (int i = 0; i < mAuxNet.NodeCnt(); i++)
			{
				int node = nodes[i];
				mDistance[node] += mDistanceBefore[node];
			}
			memcpy(mDistanceBefore, mDistance, sizeof(COST_TYPE)*mAuxNet.NodeCnt());
			/*
			cout << "DEBUG : distance after adjustment" << endl;
			nodes = mAuxNet.Nodes();
			for (int i = 0; i < mAuxNet.NodeCnt(); i++)
			{
				int node = nodes[i];
				cout << mDistance[node] << "\t";
			}
			cout << endl;
			*/


			// cout << "DEBUG : Reassign and adjust the cost" << endl;
			// Reassign and adjust the cost
			// Mask the edges that are satuated.
			mAuxNet.MaskAll(false);
			int mask_cnt = 0;
			const int* edges = mAuxNet.Edges();
			for (int i = 0; i < mAuxNet.EdgeCnt(); i++)
			{
				int edge = edges[i];
				if (!mbAux[edge] && mAuxFlow[edge] == mAuxCap[edge] ||
					 mbAux[edge] && 0 == mAuxCap[edge])
					mMaskBuf[mask_cnt++] = edge;
				
				// TRICK : To make the elements in mAuxCost always nonnegative
				int from_node = mAuxNet.FromNode(edge);
				int to_node = mAuxNet.ToNode(edge);
				mAuxCost[edge] = mCostBack[edge] + mDistance[from_node] - mDistance[to_node];

#ifdef DEBUG2
				cout << "DEBUG2 " << edge << " (" << from_node << "," << to_node << ") " 
					<< mAuxNet.IsEdgeMasked(edge) << "  " 
					<< mCostBack[edge] << "->" << mAuxCost[edge]
					<< "  flow = " << mAuxFlow[edge] << " " << "cap = " << mAuxCap[edge]
				    << "  dist[" << from_node << "] = " << mDistance[from_node]
				    << "  dist[" << to_node << "] = " << mDistance[to_node]
				    << "  costback = " << mCostBack[edge] << endl;
#endif
			}

			mAuxNet.MaskEdge(mMaskBuf, mask_cnt);
		}
	}

	return b_exist;
}		/* -----  end of method GraphAlgorithm_BusackerGowen<COST_TYPE>::OptimalFlow  ----- */



/*
 *--------------------------------------------------------------------------------------
 *        Class:  GraphAlgorithm_BusackerGowen
 *       Method:  GetFlow
 *  Description:  After the calculation is done, call this function get the flow on each
 *                edge.
 *        Param:  flow[out]  :  An array indexed by the internal edge ID of original graph
 *                              to store the flow on each edge.
 *       Return:
 *--------------------------------------------------------------------------------------
 */
	template < class COST_TYPE >
	void 
GraphAlgorithm_BusackerGowen<COST_TYPE>::GetFlow (int* flow)
{
	// Assign flow value for output
	const int* edges = mAuxNet.Edges();
	for (int i = 0; i < mAuxNet.EdgeCnt(false); i++)
	{
		int edge = edges[i];
		if (!mbAux[edge])
			flow[mAuxNet.GetEdgeExID(edge)] = mAuxFlow[edge];
#ifdef DEBUG2
		cout << "DEBUG2: flow(" << mAuxNet.FromNode(edge) << "," << mAuxNet.ToNode(edge) << ") = " << mAuxFlow[edge] << endl;
#endif
	}
	return ;
}		/* -----  end of method GraphAlgorithm_BusackerGowen<COST_TYPE>::GetFlow  ----- */


#endif


