/*
 * =====================================================================================
 *
 *       Filename:  Graph.h
 *
 *    Description:  The definition of class Graph
 *
 *        Version:  1.0
 *        Created:  03/19/2008 09:16:29 PM
 *       Revision:  11/23/2008 by feeldead
 *       Compiler:  g++
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef Graph_h
#define Graph_h

#include <set>
#include <map>
#include <vector>
#include <list>

using namespace std;

#include "Utility.hpp"

class GraphReader;
class GraphWriter;

/*
 * =====================================================================================
 *        Class:  Graph
 *  Description:  Graph provides the basic organization of a graph, based on which some
 *                algorithms would be implemented. Class graph is a much abstract 
 *                description of relations. That means that the functionalilties
 *                of connecting to the out world (for example, reading from a file to 
 *                create a graph, writting a graph to a file, or mapping the external 
 *                IDs to the inner IDs of nodes and edges) are not provided here. Class
 *                GraphEx provides those functionalilties.
 * =====================================================================================
 */
class Graph
{
	public:
		enum direction_tag
		{
			direction_out,
			direction_in,
			direction_io
		};


	public:
		/* ====================  LIFECYCLE     ======================================= */
		Graph (bool directed = true);                             /* constructor      */

		Graph ( const Graph &other );         /* copy constructor */

		virtual ~Graph ();                            /* destructor       */

		/* ====================  OPERATORS     ======================================= */
		const Graph& operator = ( const Graph &other ); /* assignment operator */

		/*
		 * Destroy
		 * Destroy the graph by release all the allocated space. It will be called
		 * whenever you plan to create a new graph.
		 */
		virtual void Destroy();

		/* ====================  OPERATIONS    ======================================= */
		/*
		 * BeginAddNodeOrEdge
		 * Call this function before you plan to create a new graph by adding nodes or
		 * edges
		 */
		virtual void BeginAddNodeOrEdge();

		/*
		 * EndAddNodeOrEdge
		 * Call this function after you have finished adding nodes and edges
		 */
		virtual void EndAddNodeOrEdge();

		/*
		 * AddNode
		 * Add a node to the graph, it will return an allocated node's internal ID
		 * Call this function after BeginAddNodeOrEdge is called
		 */
		int AddNode ();

		/*
		 * AddEdge
		 * Add an edge to the graph, it will return an allocated edge's internal ID
		 * Call this function after BeginAddNodeOrEdge is called. This ID starts 
		 * from 0 and increases by 1 each time.
		 */
		int AddEdge (int fromNode, int toNode);

		/*
		 * MaskAll
		 * Mask or unmask all the nodes and edges. This function is faster than calling
		 * MaskNode or MaskEdge multiple times
		 */
		void MaskAll (bool mask);

		/*
		 * MaskAllEdges
		 * Mask or unmask all the edges. This function is faster than calling
		 * MaskEdge multiple times
		 */
		void MaskAllEdges (bool mask);

		/*
		 * MaskNode
		 * Mask or unmask a node specified by its internal ID. You can decide whether
		 * to mask all the adjacent edges of this node by specifying maskEdges = true
		 * (for mask) or maskEdges = false (for mask the node only)
		 *
		 * The more complicated version of MaskNode (MaskEdge) takes an array of
		 * internal ID of nodes (edges) as a parameter to prevent it is called
		 * in a looping over all the nodes (edges) in the graph by 
		 * 'const int* nodes = Graph::Nodes()' or
		 * 'const int* edges = Graph::Edges()'
		 * Because the contents in 'nodes' or 'edges' will be changed by Graph::MaskNode.
		 */
		void MaskNode (const int* nodes, int len, bool mask = true, bool maskEdges = true);
		void MaskNode (const vector<int>& nodes, bool mask = true, bool maskEdges = true);
		
		/*
		 * MaskEdge
		 * Mask or unmask an edge specifyed by its internal ID.
		 */
		void MaskEdge (const int* edges, int len, bool mask = true);
		void MaskEdge (const vector<int>& edges, bool mask = true);


		/*
		 * SetDirected
		 */
		void SetDirected(bool b_directed){mbDirected = b_directed;}

		/*
		 * AllowSelfLoop
		 */
		void AllowSelfLoop (bool allowed){mbSelfLoopAllowed = allowed;}

		/*
		 * EnableMask
		 */
		void EnableMask(bool mask){mbEnableMask = mask;}

		/*
		 * Check
		 * For debug
		 */
		void Check();

		/* 
		 * Write 
		 * Write the graph to the writer 
		 */
		virtual void Write (GraphWriter* pWriter);

		/* ====================  ACCESS        ======================================= */
		/* NOTE : Do NOT call the following functions before EndAddNodeOrEdge is called.
		 * Or else the return values are meanlingless or it will cause segementation fault
		 */
		/*
		 * EdgeID
		 * Get the internal ID of an edge specified by its start node and end node.
		 * This function is influenced by whether the graph is directed or not
		 */
		int EdgeID (int fromNode, int toNode);

		/* The following method simply return the corresponding pointers 
		 * of internal data structures
		 */
		inline const int* Nodes () {return mpNodes;}

		/*
		 * In many cases, you want to create properties indexed by node ID. Then you should
		 * get the number of all the nodes in the graph instead of only those unmasked.
		 */
		inline const int  NodeCnt (bool mask = true) {return (mask ? mNodeCnt_Unmasked : mNodeCnt);}
		inline const int  OutDegree (int node) {return (mpOutDeg ? (mbEnableMask ? mpOutDeg_Unmasked[node] : mpOutDeg[node]) : 0);}
		inline const int  InDegree (int node) {return (mpInDeg ? (mbEnableMask ? mpInDeg_Unmasked[node] : mpInDeg[node]) : 0);}
		inline const int  IoDegree (int node) {return OutDegree(node) + InDegree(node);}
		inline const int* OutEdges (int node) {return (mpOutEdges ? mpOutEdges[node] : 0);}
		inline const int* InEdges (int node) {return (mpInEdges ? mpInEdges[node] : 0);}
		inline const int* Edges () {return mpEdges;}

		/*
		 * The same as NodeCnt
		 */
		inline const int  EdgeCnt (bool mask = true) {return (mask ? mEdgeCnt_Unmasked : mEdgeCnt);}
		inline const int  FromNode (int edge) {return (mpFromNode ? mpFromNode[edge] : -1);}
		inline const int  ToNode (int edge) {return (mpToNode ? mpToNode[edge] : -1);}

		/* Copy data into specified arrayes. The following methods are influenced by the mbEnableMask
		 */
		void AdjEdges(int node, list<int>& edges, direction_tag tag = direction_io);
		void AdjEdges(int node, vector<int>& edges, direction_tag tag = direction_io);
		void Neighbors(int node, list<int>& neigh, direction_tag tag = direction_io);
		void Neighbors(int node, set<int>& neigh, direction_tag tag = direction_io);
		void Neighbors(int node, vector<int>& neigh, direction_tag tag = direction_io);
		void Neighbors(const set<int>& nodes, set<int>& neigh, direction_tag tag = direction_io);
		void InducedEdges(const set<int>& nodes, set<int>& edges);

		/* ====================  INQUIRY       ======================================= */
		inline bool IsNodeMasked (int node){return (mpNodeMask ? mpNodeMask[node] : false);}
		inline bool IsEdgeMasked (int edge){return (mpEdgeMask ? mpEdgeMask[edge] : false);}
		inline bool IsNode (int node){return (node < mNodeCnt);}
		inline bool IsEdge (int fromNode, int toNode){return -1 != EdgeID(fromNode, toNode);}
		bool IsChanged () {return true;}
		bool IsDirected () {return mbDirected;}
		bool IsSelfLoopAllowed() {return mbSelfLoopAllowed;}
		bool IsMaskEnabled() {return mbEnableMask;}

		/* ====================  DATA MEMBERS  ======================================= */

	private:

		/*
		 * MaskNode
		 * Mask or unmask a node specified by its internal ID. You can decide whether
		 * to mask all the adjacent edges of this node by specifying maskEdges = true
		 * (for mask) or maskEdges = false (for mask the node only)
		 *
		 * CRITICAL : Do NOT call this function when you are looping over all
		 * the nodes in the graph by 
		 * 'const int* nodes = Graph::Nodes()' or
		 * 'const int* edges = Graph::Edges()'
		 * Because the contents in 'nodes' or 'edges' will be changed by Graph::MaskNode.
		 */
		void MaskNode (int node, bool mask = true, bool maskEdges = true);
		
		/*
		 * MaskEdge
		 * Mask or unmask an edge specifyed by its internal ID.
		 *
		 * CRITICAL : Do NOT call this function when you are looping over all
		 * the nodes in the graph by 
		 * 'const int* nodes = Graph::Nodes()' or
		 * 'const int* edges = Graph::Edges()'
		 * Because the contents in 'nodes' or 'edges' will be changed by Graph::MaskEdge.
		 */
		void MaskEdge (int edge, bool mask = true);

		void InitDataMembers();
		void MaskEdge_OutEdgeList(int edge, bool mask);
		void MaskEdge_InEdgeList(int edge, bool mask);

		// ==================== Node related begin =========================================
		// Number of nodes of the graph
		int                mNodeCnt;

		// when you add node on an graph created before, it is useful
		int 			   mNodeCnt_saved;

		// Number of unmasked nodes of the graph
		int                mNodeCnt_Unmasked;


		/*
		 * The IDs of nodes are [0..mNodeCnt-1]. However, the IDs may be stored
		 * in an shuffled order to ensure that the masked and unmaksed nodes
		 * are divided into two nonoverlapping section in the array.
		 * Note, all the properties of nodes are indexed
		 * node ID. For example, mpOutDeg[i] is the out degree of node i.
		 * mpNodes[0..mNodeCnt_Unmasked-1] stores nodes that have not been masked
		 * mpNodes[mNodeCnt_Unmasked, mNodeCnt-1] stores nodes that have been masked
		 */
		int*               mpNodes;
		int*               mpNodesPos;

		bool*              mpNodeMask;

		int*               mpOutDeg;
		int*               mpOutDeg_Unmasked;
		int**              mpOutEdges;

		int*               mpInDeg;
		int*               mpInDeg_Unmasked;
		int**              mpInEdges;

		// ==================== Node related end =========================================


		// ==================== Edge related begin =========================================
		int                mEdgeCnt;
		int                mEdgeCnt_saved;
		int                mEdgeCnt_Unmasked;

		/*
		 * The IDs of edges are [0..mEdgeCnt-1]. However, the IDs may be stored
		 * in an shuffled order as that of mpNodes
		 * mpEdges[0..mEdgeCnt_Unmasked-1] stores edges that have not been masked
		 * mpEdges[mEdgeCnt_Unmasked, mEdgeCnt-1] stores edges that have been masked
		 */
		int*               mpEdges;
		int*               mpEdgePos;
		int*               mpFromNode;
		int*               mpToNode;
		bool*              mpEdgeMask;

		/*
		 * mpFromNodePos[i] stores the possition of edge i stored in the out edge
		 * array of the start node of edge i. Exactly, 
		 * mpOutEdges[mpFromNode[i]][mpFromNodePos[i]] = i;
		 */
		int*               mpFromNodePos;

		// Similiar to mpFromNodePos.
		int*               mpToNodePos;

		// Map the combined internal IDs of the from and to nodes
		// of the edge to the internal ID of the edge
		map<int64, int>           mExistingEdges;
		// ==================== Edge related end =========================================

		// Allow self loop or not? This parameter influence the creation of
		// the graph.
		bool               mbSelfLoopAllowed;

		// The graph is directed or not? This parameter influence the creation
		// and behaviours of some of the methods.
		bool               mbDirected;

		// Is the mask enableed not not? This parameter influence the behaviours 
		// of some of the methods.
		bool               mbEnableMask;


}; /* -----  end of class  Graph  ----- */

#endif
