#ifndef ISOFORM_OPT_H
///Define this macro to prevent from including this header file more than once.
#define ISOFORM_OPT_H

#include "math_utils.h"
#include "exon_junction_extractor.h"
//#include "gsnake.h"

const int max_iteration = 10000;

double sum_matrix(const vector<vector<double> > &matrix) {
	double result = 0;
	for (int i = 0; i < (int)matrix.size(); i++)
		for (int j = 0; j < (int)matrix[i].size(); j++)
			result += matrix[i][j];
	return result;
}

/*vector<vector<double> > inv_matrix(const vector<vector<double> > &matrix) {
	int m = (int)matrix.size(), n = (int)matrix[0].size();
	MATRIX input, *output;
	input.init((short)m, (short)n);
	for (int i = 0; i < m; i++)
		for (int j = 0; j < n; j++)
			input.put(i, j, matrix[i][j]);
	output = input.inverse();
	vector<vector<double> > result;
	result.resize(m);
	for (int i = 0; i < m; i++) {
		result[i].resize(m);
		for (int j = 0; j < m; j++)
			result[i][j] = output->get(i,j);
	}
	delete output;
	return result;
}*/

class isoform_opt{
public:
	bool verbose;
	int M; //number of subexon_sets
	int K; //number of isoforms
	double NN; //total number of mappable reads
	vector<double> N; //subexon_set read counts
	vector<double> L; //subexon_set lengths
	vector<vector<bool> > A; //subexon_set_indicator matrix; A[i][j] is true if isoform i contain subexon_set j;

	isoform_opt() {verbose = false;}

	double log_likelihood(const vector<double> &X) {
		double result = 0;
		double temp1 = 0;	
		for (int i = 0; i < M; i++) {
			double temp2 = 0;
			for (int j = 0; j < K; j++) {
				if (A[j][i]) temp2 += X[j];
			}
			temp1 += temp2 * L[i];
		}
		result -= temp1 * NN;
		temp1 = 0;
		for (int i = 0; i < M; i++) {
			double temp2 = 0;
			for (int j = 0; j < K; j++) {
				if (A[j][i]) temp2 += X[j];
			}
			temp1 += log(temp2 + epsilon) * N[i];
		}
		result += temp1;
		return result;
	}

	double grad_log_likelihood(const vector<double> &X, int k) {
		double result = 0;
		double temp1 = 0;	
		for (int i = 0; i < M; i++) {
			if (A[k][i]) temp1 += L[i];
		}
		result -= temp1 * NN;
		temp1 = 0;
		for (int i = 0; i < M; i++) {
			if (!A[k][i]) continue;
			double temp2 = 0;
			for (int j = 0; j < K; j++) {
				if (A[j][i]) temp2 += X[j];
			}
			temp1 += N[i] / (temp2 + epsilon);
		}
		result += temp1;
		return result;
	}

	vector<double> grad_log_likelihood(const vector<double> &X) {
		vector<double> grad;
		grad.resize(K);
		for (int i = 0; i < K; i++) {
			grad[i] = grad_log_likelihood(X, i);
		}
		return grad;
	}

	vector<vector<double> > hessian_log_likelihood(const vector<double> &X) {
		vector<vector<double> > hessian;
		hessian.resize(K);
		for (int i = 0; i < K; i++) hessian[i].resize(K);
		vector<double> temp1;
		temp1.resize(M);
		for (int i = 0; i < M; i++) {
			double temp2 = 0;
			for (int j = 0; j < K; j++) {
				if (A[j][i]) temp2 += X[j];
			}
			temp1[i] = N[i] / (temp2 + epsilon) / (temp2 + epsilon);
		}
		for (int k = 0; k < K; k++) {
			for (int l = 0; l < K; l++) {
				hessian[k][l] = 0;
				for (int i = 0; i < M; i++) {
					if (A[k][i] && A[l][i]) hessian[k][l] -= temp1[i];
				}
				if (k == l) hessian[k][l] -= 1e-6;
			}
		}
		return hessian;
	}

	vector<vector<double> > obs_fisher(const vector<double> &X) {
		vector<vector<double> > temp = hessian_log_likelihood(X);
		for (int i = 0; i < K; i++)
			for (int j = 0; j < K; j++)
				temp[i][j] = -temp[i][j];
		return temp;
	}

/*	vector<vector<double> > inv_obs_fisher(const vector<double> &X) {
		return inv_matrix(obs_fisher(X));
	}*/
	
	vector<vector<double> > fisher(const vector<double> &X) {
		vector<vector<double> > fisher;
		fisher.resize(K);
		for (int i = 0; i < K; i++) fisher[i].resize(K);
		vector<double> temp1;
		temp1.resize(M);
		for (int i = 0; i < M; i++) {
			double temp2 = 0;
			for (int j = 0; j < K; j++) {
				if (A[j][i]) temp2 += X[j];
			}
			temp1[i] = NN * L[i] / (temp2 + epsilon);
		}
		for (int k = 0; k < K; k++) {
			for (int l = 0; l < K; l++) {
				fisher[k][l] = 0;
				for (int i = 0; i < M; i++) {
					if (A[k][i] && A[l][i]) fisher[k][l] += temp1[i];
				}
				if (k == l) fisher[k][l] += 1e-6;
			}
		}
		return fisher;
	}

/*	vector<vector<double> > inv_fisher(const vector<double> &X) {
		return inv_matrix(fisher(X));
	}*/

	double fmax_coord(vector<double> &X, int k) {
		double f_value = log_likelihood(X);
		double grad = grad_log_likelihood(X, k);
		if (grad > 0) grad = 1; else if (grad < 0) grad = -1; else return f_value;
		double x_old = X[k];
		double step = 1e6;
		int iteration = 0;
		while (true) {
			X[k] = x_old + step * grad;
			if (X[k] < 0) X[k] = 0;
			double new_f_value = log_likelihood(X);
			if (new_f_value > f_value) return new_f_value;
			if (step > epsilon) step /= 2; else break;
			if (iteration < max_iteration) iteration++; else break;
		}
		//if (verbose && step <= epsilon) cout << "warning: step <= epsilon\n";
		if (iteration >= max_iteration) cout << "warning: iteration >= max_iteration\n";
		X[k] = x_old;
		return f_value;
	}

	double fmax_coord(vector<double> &X) {
		double f_value = log_likelihood(X);
		int iteration = 0;
		while (true) {
			if (verbose) {
				cout << "iteration: " << iteration << " value: " << f_value << " X: ";
				for (int i = 0; i < K; i++) cout << X[i] << ",";
				cout << endl;
			}
			for (int i = 0; i < K; i++) {
				fmax_coord(X, i);
			}
			double new_f_value = log_likelihood(X);
			if (fabs(f_value - new_f_value) < epsilon) {
				if (verbose) cout << "number of iteration: " << iteration << endl;
				return new_f_value;
			}
			f_value = new_f_value;
			if (iteration < max_iteration) iteration++; else break;
		}
		if (iteration >= max_iteration) cout << "warning: iteration >= max_iteration\n";
		return f_value;
	}

	void test() {
		M = 3;
		K = 2;
		NN = 1;
		N.resize(M);
		N[0] = 100;
		N[1] = 700;
		N[2] = 500;
		L.resize(M);
		L[0] = 1;
		L[1] = 1;
		L[2] = 1;
		A.resize(K);
		A[0].resize(M);
		A[0][0] = 1;
		A[0][1] = 1;
		A[0][2] = 0;
		A[1].resize(M);
		A[1][0] = 0;
		A[1][1] = 1;
		A[1][2] = 1;
		verbose = true;
		vector<double> X;
		X.resize(2);
		X[0] = 1;
		X[1] = 1;
		fmax_coord(X);
		cout << "X: " << X[0] << "\t" << X[1] << endl;
		vector<vector<double> > H = hessian_log_likelihood(X);
		cout << "H: ";
		cout << "\t" << H[0][0] << "\t" << H[0][1] << endl;
		cout << "\t" << H[1][0] << "\t" << H[1][1] << endl;	
		vector<vector<double> > OF = obs_fisher(X);
		cout << "OF: ";
		cout << "\t" << OF[0][0] << "\t" << OF[0][1] << endl;
		cout << "\t" << OF[1][0] << "\t" << OF[1][1] << endl;	
		vector<vector<double> > F = fisher(X);
		cout << "F: ";
		cout << "\t" << F[0][0] << "\t" << F[0][1] << endl;
		cout << "\t" << F[1][0] << "\t" << F[1][1] << endl;	
/*		vector<vector<double> > IOF = inv_obs_fisher(X);
		cout << "IOF: ";
		cout << "\t" << IOF[0][0] << "\t" << IOF[0][1] << endl;
		cout << "\t" << IOF[1][0] << "\t" << IOF[1][1] << endl;	
		vector<vector<double> > IF = inv_fisher(X);
		cout << "IF: ";
		cout << "\t" << IF[0][0] << "\t" << IF[0][1] << endl;
		cout << "\t" << IF[1][0] << "\t" << IF[1][1] << endl;	*/
	}
};

#endif // ISOFORM_OPT_H
