// =====================================================================================
// 
//       Filename:  ReleaseInterface.h
// 
//    Description:  The interface for the final user
// 
//        Version:  1.0
//        Created:  11/12/2009 08:57:00 AM
//       Revision:  none
//       Compiler:  g++
// 
//         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
//        Company:  THU
// 
// =====================================================================================

#ifndef ReleaseInterface_H 
#define ReleaseInterface_H

#include <string>
#include <string.h>
#include <iostream>
#include "DataProcessor.h"
#include <vector>
#include "InstanceWriter.h"
#include "InstanceReader.h"
#include "IsoInfer.h"
#include "ResultSummary.h"
#include "ExpressionCalculator.h"

#include "ToolBase.h"

using namespace std;

/*
 * =====================================================================================
 *        Class:  ReleaseInterface
 *  Description:  This class provides methods to operates and processes the data.
 * =====================================================================================
 */
class ReleaseInterface : public ToolBase
{
	public:
		/* ====================  LIFECYCLE     ======================================= */
		ReleaseInterface (){};                             /* constructor */

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  ReleaseInterface 
		 *       Method:  Help
		 *  Description:  
		 *        Param:  
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		virtual
		void
		Help()
		{
			cout << endl;
			cout << "      ============================================================================ " << endl;
			cout << "          IsoInfer : Infer isoforms based on single-end, pairted-end, TSS/PAS and " << endl;
			cout << "                     exon-intron boundary information. " << endl;
			cout <<                                                   endl;
			cout << "          Author   : Jianxing Feng. feeldead@gmail.com" << endl;
			cout << "            Date   : Thu Feb 25 18:41:11 PST 2010" << endl;
			cout << "          Version  : 0.5                              " << endl;
			cout << "      ============================================================================ " << endl;
			cout <<                                                   endl;
			cout << "      USAGE:   IsoInfer <JOB> <PARAMETERS>          " << endl;
			cout <<                                endl;
			cout << "      Manual:  www.cs.ucr.edu/~jianxing/IsoInfer.html " << endl;
			cout <<                                endl;
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  ReleaseInterface 
		 *       Method:  Main
		 *  Description:  
		 *        Param:  
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		virtual
		int 
		Main(int argc, char* argv[], int startArg)
		{
			ostream* p_output;

			// Jobs
			bool b_help = false;
			bool b_predict = false;
			bool b_ext_junc_ref = false;
			bool b_gen_instance = false;
			// Interna jobs
			bool b_cal_exp = false;
			bool b_uniq_exp = false;
			bool b_rand_exp = false;
			bool b_rand_reads = false;
			bool b_write_known = false;
			bool b_high_isoform = false;
			bool b_group_isoform = false;
			bool b_app_read_info = false;
			bool b_v05_2_v04= false;
			bool b_compare = false;
			bool b_rel_diff = false;
			// Jobs end

			 
			// parameters
			string refseq_file = "";
			string read_info_file = "";
			string output_file = "";
			bool b_strand_specific = false;
			string ins_file = "";
			int start_pos_of_first_nt = 0;
			string bound_file = "";
			string grange_file = "";
			string tsspas_file = "";
			bool b_pe_enable = false;
			bool b_se_enable = true;
			double min_exp = 1;
			int min_dup = 1;
			int part_size = 7;
			double conf_level = 0.05;
			// Internal parametes
			int output_format = 1;
			int rand_type = 0;
			int min_iso = 8;
			int iso_scale = 9;
			bool b_use_provided_exons = false;
			double noise_level = 0;
			int cal_exp_method= 2;
			bool b_bs2 = true;
			string pred_file = "";
			string ben_file = "";
			int handle_read_method = 0;
			double true_scale = 1;
			// Parameters end

			// Begin parsing the jobs and parameters
			for (int i = startArg; i < argc; i++)
			{
				if (strcmp(argv[i], "-h") == 0)
				{
					Help();
					return 0;
				}

				if (strcmp(argv[i], "-ext_junc_ref") == 0)
					b_ext_junc_ref = true;
				else if (strcmp(argv[i], "-gen_instance") == 0)
					b_gen_instance = true;
				else if (strcmp(argv[i], "-predict") == 0)
					b_predict = true;

				// Internal jobs start
				else if (strcmp(argv[i], "-cal_exp") == 0)
					b_cal_exp= true;
				else if (strcmp(argv[i], "-uniq_exp") == 0)
					b_uniq_exp= true;
				else if (strcmp(argv[i], "-rand_exp") == 0)
					b_rand_exp= true;
				else if (strcmp(argv[i], "-rand_reads") == 0)
					b_rand_reads = true;
				else if (strcmp(argv[i], "-write_known") == 0)
					b_write_known = true;
				else if (strcmp(argv[i], "-high_isoform") == 0)
					b_high_isoform= true;
				else if (strcmp(argv[i], "-group_isoform") == 0)
					b_group_isoform= true;
				else if (strcmp(argv[i], "-app_read_info") == 0)
					b_app_read_info= true;
				else if (strcmp(argv[i], "-v05_2_v04") == 0)
					b_v05_2_v04 = true;
				else if (strcmp(argv[i], "-compare") == 0)
					b_compare = true;
				else if (strcmp(argv[i], "-rel_diff") == 0)
					b_rel_diff = true;
				// Internal jobs end

				else if (strcmp(argv[i], "-ref") == 0)
					refseq_file = argv[++i];
				else if (strcmp(argv[i], "-read_info") == 0)
					read_info_file = argv[++i];
				else if (strcmp(argv[i], "-o") == 0)
					output_file = argv[++i];
				else if (strcmp(argv[i], "-s") == 0)
					b_strand_specific = (argv[++i][0] == 'T');
				else if (strcmp(argv[i], "-ins") == 0)
					ins_file = argv[++i];
				else if (strcmp(argv[i], "-rstart") == 0)
					start_pos_of_first_nt = atoi(argv[++i]);
				else if (strcmp(argv[i], "-bound") == 0)
					bound_file = argv[++i];
				else if (strcmp(argv[i], "-grange") == 0)
					grange_file = argv[++i];
				else if (strcmp(argv[i], "-tsspas") == 0)
					tsspas_file = argv[++i];
				else if (strcmp(argv[i], "-bpe") == 0)
					b_pe_enable = (argv[++i][0] == 'T');
				else if (strcmp(argv[i], "-bse") == 0)
					b_se_enable = (argv[++i][0] == 'T');
				else if (strcmp(argv[i], "-min_exp") == 0)
					min_exp = atof(argv[++i]);
				else if (strcmp(argv[i], "-min_dup") == 0)
					min_dup = atoi(argv[++i]);
				else if (strcmp(argv[i], "-ps") == 0)
					part_size = atoi(argv[++i]);
				else if (strcmp(argv[i], "-noise") == 0)
					noise_level = atof(argv[++i]);
				else if (strcmp(argv[i], "-conf_level") == 0)
					conf_level = atof(argv[++i]);

				// Internal options begin
				else if (strcmp(argv[i], "-oformat") == 0)
					output_format = atoi(argv[++i]);
				else if (strcmp(argv[i], "-rand_type") == 0)
					rand_type = atoi(argv[++i]);
				else if (strcmp(argv[i], "-min_iso") == 0)
					min_iso = atoi(argv[++i]);
				else if (strcmp(argv[i], "-iso_scale") == 0)
					iso_scale = atoi(argv[++i]);
				else if (strcmp(argv[i], "-use_provided_exon") == 0)
					b_use_provided_exons = (argv[++i][0] == 'T');
				else if (strcmp(argv[i], "-cal_exp_method") == 0)
					cal_exp_method = atoi(argv[++i]);
				else if (strcmp(argv[i], "-bs2") == 0)
					b_bs2 = (argv[++i][0] == 'T');
				else if (strcmp(argv[i], "-pred") == 0)
					pred_file = argv[++i];
				else if (strcmp(argv[i], "-ben") == 0)
					ben_file = argv[++i];
				else if (strcmp(argv[i], "-handle_read_method") == 0)
					handle_read_method = atoi(argv[++i]);
				else if (strcmp(argv[i], "-true_scale") == 0)
					true_scale = atoi(argv[++i]);
				// Internal options end
				
				else 
				{
					Help();
					cerr << "Wrong parameter " << argv[i] << endl;
					exit(0);
				}
			}

			if ( !b_predict && 
				 !b_ext_junc_ref  && 
				 !b_gen_instance  && 
				 !b_cal_exp && 
				 !b_uniq_exp &&
				 !b_rand_exp && 
				 !b_write_known && 
				 !b_high_isoform && 
				 !b_group_isoform && 
				 !b_rand_reads && 
				 !b_app_read_info &&
				 !b_v05_2_v04 &&
				 !b_compare &&
				 !b_rel_diff)
			{

				Help(); ArgMissing("Job"); exit(0);
			}

			// Prepare output
			if (output_file != "")
			{
				p_output = new ofstream(output_file.data(), ios::out);
				if (!((ofstream*)p_output)->is_open())
				{
					cerr << "File " << output_file.data() << " can not be opened" << endl;
					return false;
				}
			}
			else
				p_output = &cout;

			DataProcessor dp;
			vector<ReadInfoBase*> read_infos;
			vector<string> mapping_files;
			if (read_info_file != "")
				dp.CreateReadInfo(read_info_file, read_infos, mapping_files);
			dp.mbStrandSpecific = b_strand_specific;
			dp.mpOutput = p_output;

			ExpEstimator lp_solver;
			ExpEstimator::ExpMethod exp_method;
			switch (cal_exp_method)
			{
				case 0:
					exp_method = ExpEstimator::EXP_METHOD_UNIQ; break;
				case 1:
					exp_method = ExpEstimator::EXP_METHOD_LP; break;
				case 2:
					exp_method = ExpEstimator::EXP_METHOD_QP; break;
				case 3:
					exp_method = ExpEstimator::EXP_METHOD_MLIS; break;
				default:
					exp_method = ExpEstimator::EXP_METHOD_QP;
			}
			lp_solver.SetMethod(exp_method);

			if (b_predict)
			{
				if (ins_file == "") {Help(); ArgMissing("-ins"); exit(0);}
				if (read_info_file == "") { Help(); ArgMissing("-read_info"); exit(0);}

				for (unsigned i = 0; i < read_infos.size(); ++i)
					read_infos[i]->mMinExpLevel = min_exp;

				IsoInfer iso_infer(&lp_solver, p_output);
				iso_infer.SetConfidencelevel(conf_level);
				iso_infer.EnableSE(b_se_enable);
				iso_infer.EnableStepII(b_bs2);
				iso_infer.SetMinEffectivePartCombDup(min_dup);
				iso_infer.SetPartitionSize(part_size);

				ResultSummary summary(&iso_infer, p_output);
				summary.SetMinIsoCnt(min_iso);
				summary.SetIsoCntScale(iso_scale);
				summary.SetOutputFormat(output_format); 

				InstanceWriter writer(p_output);
				InstanceReader reader(ins_file, read_infos);
				reader.SetHandler(&summary);

				reader.Initialize();
				reader.Generate();
				reader.CleanUp();
			}
			else if (b_ext_junc_ref)
			{
				if (bound_file == "") {Help(); ArgMissing("-bound"); exit(0);}
				if (grange_file == "") { Help(); ArgMissing("-grange"); exit(0);}
				if (tsspas_file == "") { Help(); ArgMissing("-tsspas"); exit(0);}
				if (read_info_file == "") { Help(); ArgMissing("-read_info"); exit(0);}

				map_str2vec_gene genes;
				map_str2vec_exon exons;
				map_str2vec_int  gene_color;
				map_str2vec_int  exon_color;

				if (!dp.LoadGeneAndExon(bound_file, grange_file, tsspas_file, genes, exons)) return 1;
				dp.GroupGeneAndExon(genes, gene_color, exons, exon_color);

				// Take the information of the first read info
				dp.ExtractJunctionRef(exons, exon_color, genes, gene_color, refseq_file, 
						start_pos_of_first_nt, read_infos[0]->ReadLen(), read_infos[0]->mCrossStrength);
			}
			else if (b_gen_instance)
			{
				if (bound_file == "") {Help(); ArgMissing("-bound"); exit(0);}
				if (grange_file == "") { Help(); ArgMissing("-grange"); exit(0);}
				if (tsspas_file == "") { Help(); ArgMissing("-tsspas"); exit(0);}
				if (read_info_file == "") { Help(); ArgMissing("-read_info"); exit(0);}

				map_str2vec_gene genes;
				map_str2vec_exon exons;
				map_str2vec_int  gene_color;
				map_str2vec_int  exon_color;

				if (!dp.LoadGeneAndExon(bound_file, grange_file, tsspas_file, genes, exons)) return 1;
				dp.GroupGeneAndExon(genes, gene_color, exons, exon_color);

				vector<Instance> all_instances;
				dp.ExtractInstances(genes, gene_color, exons, exon_color, all_instances);

				for (unsigned i = 0; i < read_infos.size(); ++i)
					dp.LoadShortReads(mapping_files[i], exons, read_infos[i], all_instances);

				InstanceWriter writer(p_output);
				writer.Initialize();
				for (unsigned i = 0; i < all_instances.size(); i++)
				{
					int seg_cnt = dp.RemoveIntrons(all_instances[i], b_use_provided_exons, noise_level);
					if (seg_cnt > 0) writer.OnInstance(all_instances[i]);
				}
				writer.CleanUp();
			}
			else if (b_cal_exp)
			{
				InstanceWriter writer(p_output);
				writer.SetConfidencelevel(conf_level);
				writer.SetWriteFormat(InstanceWriter::WRITE_EXP);

				ExpressionCalculator exp_calculator(&lp_solver, p_output);
				exp_calculator.SetHandleReadMethod(handle_read_method);
				exp_calculator.AddHandler(&writer);

				InstanceReader reader(ins_file, read_infos);
				reader.SetHandler(&exp_calculator);
				reader.Initialize();
				reader.Generate();
				reader.CleanUp();
			}
			else if (b_rand_exp)
			{
				RandomExpReadAssignerIM semi_rand_gen(p_output);
				semi_rand_gen.SetRandExpType(rand_type);

				InstanceWriter writer(p_output);
				semi_rand_gen.AddHandler(&writer);

				InstanceReader reader(ins_file, read_infos);
				reader.SetHandler(&semi_rand_gen);
				reader.Initialize();
				reader.Generate();
				reader.CleanUp();
			}
			else if (b_rand_reads)
			{
				for (unsigned i = 0; i < mapping_files.size(); ++i)
				{
					ofstream output;

					output.open(mapping_files[i].data(), ios::out);
					if (!output.is_open())
					{
						cerr << "File " << mapping_files.data() << " can not be opened" << endl;
						return false;
					}

					RandomExpReadAssignerIM semi_rand_gen(&output);
					semi_rand_gen.SetReadInfo(read_infos[i]);

					InstanceReader reader(ins_file, read_infos);
					reader.SetHandler(&semi_rand_gen);

					reader.Initialize();
					reader.Generate();
					reader.CleanUp();

					output.close();
				}
			}
			else if (b_group_isoform)
			{
				if (bound_file == "") {ArgMissing("-bound"); Help();}
				if (grange_file == "") { ArgMissing("-grange"); Help(); }
				if (tsspas_file == "") { ArgMissing("-tsspas"); Help(); }

				map_str2vec_gene genes;
				map_str2vec_exon exons;
				map_str2vec_int  gene_color;
				map_str2vec_int  exon_color;

				if (!dp.LoadGeneAndExon(bound_file, grange_file, tsspas_file, genes, exons)) return 1;
				dp.GroupGeneAndExon(genes, gene_color, exons, exon_color);

				for_each_ele_in_group(iter, map_str2vec_gene, genes)
				{
					vector<Gene>& curr_genes = iter->second;
					vector<int>& curr_gene_colors = gene_color[iter->first];

					for (unsigned i = 0; i < curr_genes.size(); i++)
						(*p_output) << curr_genes[i].mName << "\t" << curr_gene_colors[i] << "\t" << i << endl;
				}
			}
			else if (b_app_read_info)
			{
				if (ins_file == "") {ArgMissing("-ins"); Help();}

				InstanceReader reader(ins_file, read_infos);
				vector<Instance> all_instances;
				reader.ReadAllInstances(all_instances);

				// Setup exons from information of all the instances
				map_str2vec_exon exons;
				for (unsigned i = 0; i < all_instances.size(); ++i)
				{
					string chr = all_instances[i].mExons[0].mChr;
					bool strand = all_instances[i].mExons[0].mStrand;
					if (b_strand_specific)
						chr += (strand ? '+' : '-');

					if (exons.find(chr) == exons.end())
					{
						vector<Exon> dummy;
						exons[chr] = dummy;
					}

					vector<Exon>& curr_exons = exons[chr];
					for (unsigned j = 0; j < all_instances[i].mExons.size(); ++j)
						curr_exons.push_back(all_instances[i].mExons[j]);
				}
				for_each_ele_in_group(iter, map_str2vec_exon, exons)
					sort(iter->second.begin(), iter->second.end());

				for (unsigned i = 0; i < read_infos.size(); ++i)
					dp.LoadShortReads(mapping_files[i], exons, read_infos[i], all_instances);

				InstanceWriter writer(p_output);
				writer.Initialize();
				for (unsigned i = 0; i < all_instances.size(); i++)
				{
					int seg_cnt = dp.RemoveIntrons(all_instances[i], b_use_provided_exons, noise_level);
					if (seg_cnt > 0) writer.OnInstance(all_instances[i]);
				}
				writer.CleanUp();
			}
			else if (b_v05_2_v04)
			{
				if (ins_file == "") {Help(); ArgMissing("-ins"); exit(0);}
				InstanceWriter writer(p_output);
				writer.SetWriteFormat(InstanceWriter::WRITE_V04);

				InstanceReader reader(ins_file, read_infos);
				reader.SetHandler(&writer);

				reader.Initialize();
				reader.Generate();
				reader.CleanUp();
			}
			else if (b_write_known)
			{
				if (ins_file == "") {Help(); ArgMissing("-ins"); exit(0);}
				//if (read_info_file == "") {Help(); ArgMissing("-read_info"); exit(0);}

				InstanceWriter writer(p_output);
				writer.SetWriteFormat(InstanceWriter::WRITE_KNOWN);

				InstanceReader reader(ins_file, read_infos);
				reader.SetHandler(&writer);
				reader.Initialize();
				reader.Generate();
				reader.CleanUp();
			}
			else if (b_compare)
			{
				if (pred_file == "") {Help(); ArgMissing("-pred"); exit(0);}
				if (ben_file == "") {Help(); ArgMissing("-ben"); exit(0);}

				ResultSummary summary(NULL, p_output);
				summary.SetMinIsoCnt(min_iso);
				summary.SetIsoCntScale(iso_scale);
				summary.PredVSBenchmark(pred_file, ben_file);
			}
			else if (b_rel_diff)
			{
				if (pred_file == "") {Help(); ArgMissing("-pred"); exit(0);}
				if (ben_file == "") {Help(); ArgMissing("-ben"); exit(0);}

				ResultSummary summary(NULL, p_output);
				summary.RelDiff(ben_file, pred_file, true_scale);
			}

			// Clean memories and handlers.
			if (output_file != "")
			{
				((ofstream*)p_output)->close();
				delete p_output;
			}

			if (read_info_file != "")
			{
				for (unsigned i = 0; i < read_infos.size(); ++i)
					delete read_infos[i];
			}
			return 0;
		}
}; /* -----  end of class ReleaseInterface  ----- */

#endif
