// =====================================================================================
// 
//       Filename:  ReadInfoBase.hpp
// 
//    Description:  The implementation of class ReadInfoBase
// 
//        Version:  1.0
//        Created:  02/05/2010 04:10:25 PM
//       Revision:  none
//       Compiler:  g++
// 
//         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
//        Company:  THU
// 
// =====================================================================================

#ifndef ReadInfoBase_H
#define ReadInfoBase_H

#include <vector>
#include <string>
#include <iostream>
#include <algorithm>
#include <cmath>
#include "Utility.hpp"
#include "Distribution.hpp"

using namespace std;

class Pattern;

/*
 * =====================================================================================
 *        Class:  ReadInfoBase
 *  Description:  This class defines the basic information for all type of reads
 * =====================================================================================
 */
class ReadInfoBase
{
	public:
		// For a read across junction, each of the involved exon must
		// contribute at least mCrossStrength bases.
		int 					mCrossStrength;
		double 					mNoiseLevel;

		// The total number of reads sampled in the whole experiments
		double 					mTotalReadCnt;

		Distribution*           mpReadLenDist;

		// The following two member variables are calculated based on the 
		// former four member variables.
		double 					mJuncLen;

		// mMinExpLevel is used in method IsInformative. This parameter should 
		// be set explicitely.
		double 					mMinExpLevel;

	public : 
		ReadInfoBase()
		{
			mpReadLenDist = 0;
			mJuncLen = -1;
			mMinExpLevel = 0;
			mCrossStrength = 1;
		}

		~ReadInfoBase()
		{
			if (mpReadLenDist) delete mpReadLenDist;
		}

		virtual void Read(istream* p_in)
		{
			(*p_in) >> mCrossStrength;
			(*p_in) >> mNoiseLevel;
			(*p_in) >> mTotalReadCnt;
			int dist_type;
			(*p_in) >> dist_type;

			if (0 == dist_type)
				mpReadLenDist = new DistributionConstant();
			else if (1 == dist_type)
				mpReadLenDist = new DistributionGaussian();
			else if (2 == dist_type)
				mpReadLenDist = new DistributionCustomize();
			else
			{
				cerr << "Error : Unrecognized distribution type" << endl;
				exit(0);
			}

			mpReadLenDist->Read(p_in);
		}

		virtual double ReadLen(gsl_rng* rng = 0)
		{
			return mpReadLenDist->GetValue(rng);
		}

		virtual double JuncLen() 
		{
			if (mJuncLen < 0)
			{
				mJuncLen = 0;
				int read_len_high = mpReadLenDist->RangeHigh();
				for (int i = mCrossStrength; i < read_len_high; ++i)
				{
					// P(X >= i + mCrossStrength)
					double prob_in_range = 1 - mpReadLenDist->ProbLessThan(i + mCrossStrength);
					mJuncLen += prob_in_range;
				}
			}
			return mJuncLen;
		}

		virtual bool IsSingleEnd() {return true;}
		virtual bool IsConstantLen() {return true;}

		// Given an isoform, the length of each segment and the index of
		// a segment, get all the informative patterns containing the 
		// specified segment and some segments before the specified 
		// segment and the virtual length of each pattern.
		virtual void InformativePatterns(const vector<bool>& isoform, 
				const vector<int>& seg_lens, vector<Pattern>& patterns, 
				vector<double>& vir_lens, int curr_seg) = 0;

		// Given an isoform, the length of each segment and the index of
		// a segment, get all the informative patterns 
		virtual void InformativePatterns(const vector<bool>& isoform, const vector<int>& seg_lens, 
				vector<Pattern>& patterns, vector<double>& vir_lens)
		{
			for (unsigned i = 0; i < isoform.size(); ++i)
				if (isoform[i]) InformativePatterns(isoform, seg_lens, patterns, vir_lens, i);
		}

		// Given the length of each segment, an isoform and a pattern, calculate
		// the virtual length of the pattern
		virtual double PatternVirtualLength(const vector<bool>& isoform,
				const vector<int>& seg_lens, const Pattern& pattern) = 0;

		// Whether a given virtual length is informative
		virtual bool IsInformative(double vir_len)
		{
			double exp_in_read_cnt = vir_len / 1000 * mMinExpLevel * mTotalReadCnt / 1000000;

			// Calculate the probability of not observing any reads
			double prob = 1 / exp(exp_in_read_cnt);

			if (prob < 0.05)
				return true;

			return false;
		}

		// Return the minimum expression level for a junction to be informative
		virtual double MinExpForJunctionToBeInformative()
		{
			double exp_in_read_cnt = log (1 / 0.05);
			return exp_in_read_cnt / (JuncLen() / 1000 * mTotalReadCnt / 1000000);
		}
};

#endif
