/*
 * =====================================================================================
 *
 *       Filename:  Utility.cc
 *
 *    Description:  The implementation of functions in namespace Utility
 *
 *        Version:  1.0
 *        Created:  05/20/2008 10:37:21 PM
 *       Revision:  none
 *       Compiler:  g++
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#include "Utility.hpp"
#include <string>
#include <algorithm>
#include <complex>

/*
 *--------------------------------------------------------------------------------------
 *   namespace:  Utility
 *      Method:  Split
 * Description:  Split a string when ever a char in the delimits is encountered
 *       Param:  delimits    :    seperate the string whenever a char in \"delimits\"
 *                                is encountered
 *               data        :    the string being seperated.
 *               escape      :    the escape character in \"data\", default value is \\
 *      Return:  The seperated strings
 *--------------------------------------------------------------------------------------
 */
vector<string> 
Utility::Split(const vector<char>& delimits, const string& data, char escape)
{
	vector<string> splitted;
	int last = 0;
	char pre = (char)(escape + 1);
	for (int i = 0; i < data.size(); i++)
	{
		bool is_delimits = false;
		for (vector<char>::const_iterator iter = delimits.begin(); iter != delimits.end(); iter++)
		{
			if (data[i] == *iter && escape != pre)
			{
				is_delimits = true;
				break;
			}
			if (escape == pre)
				pre = (char)(escape + 1);
			else
				pre = data[i];
		}
		if (!is_delimits && i == data.size() - 1)
		{
			i++;
			is_delimits = true;
		}
		if (is_delimits)
		{
			pre = (char)(escape + 1);
			string part = data.substr(last, i - last);
			int j = 0;
			for (int k = 0; k < part.size(); k++)
			{
				if (part[k] != escape || k > 0 && part[k-1] == escape)
					part[j++] = part[k];
			}
			splitted.push_back(part.substr(0, j));
			last = i + 1;
		}
	}
	return splitted;
} /* -----  end of method Utility::Split  ----- */

/*
 *--------------------------------------------------------------------------------------
 *   namespace:  Utility
 *      Method:  Split
 * Description:  Split a string when ever a delimit is encountered
 *       Param:  delimits    :    seperate the string whenever a delimit is encountered
 *               data        :    the string being seperated.
 *               escape      :    the escape character in \"data\", default value is \\
 *      Return:  The seperated strings
 *--------------------------------------------------------------------------------------
 */
vector<string> 
Utility::Split(char delimit, const string& data, char escape)
{
	vector<char> delimits;
	delimits.push_back(delimit);
	return Utility::Split(delimits, data, escape);
} /* -----  end of method Utility::Split  ----- */

string Utility::Transform2UniqueString(set<int>& IntSet)
{
	vector<int> vSet;
	vSet.assign(IntSet.begin(), IntSet.end());
	return Transform2UniqueString(vSet);
}

string Utility::Transform2UniqueString(vector<int>& IntSet)
{
	vector<int> vSet;
	vSet.assign(IntSet.begin(), IntSet.end());
	sort(vSet.begin(), vSet.end());
	char* buf = new char[IntSet.size()*sizeof(int) + 1];
	size_t i;
	for (i = 0; i < IntSet.size(); i++)
	{
		char* p = buf + i * sizeof(int);
		int j = IntSet[i];
		p[0] = Utility::Digit2Hex(char((j >> 12) & 0x000F));
		p[1] = Utility::Digit2Hex(char((j >> 8) & 0x000F));
		p[2] = Utility::Digit2Hex(char((j >> 4) & 0x000F));
		p[3] = Utility::Digit2Hex(char((j & 0x000F)));
	}
	buf[i * sizeof(int)] = 0;
	string re(buf);
	delete[] buf;
	return re;
}

char Utility::Digit2Hex(char digit)
{
	return (digit > 9) ? (digit - 10 + 'A') : (digit + '0');
}

/*
 * result = op1 - op2;
 * return the possible float calculation error
 */
double Utility::minusError(double& result, double& op1, double& op2)
{
	double re= op1 - op2;
	return max(max(abs(op1 - (re+ op2)),
			   abs(op1 - re- op2)), abs(op1 - op2 - re));
	result = re;
}

/*
* result = op1 + op2;
* return the possible float calculation error
*/
double Utility::plusError(double& result, double& op1, double& op2)
{
	double re = op1 + op2;
	return max(abs(re - op1 - op2), abs(re- op2 - op1));
	result = re;
}

/*
 * Truncate the part of the float number less than 2^-epsilon
 */
double Utility::precisionControl(double fnum, int precision)
{
	/*
	// double format:
	//  63    62 -- 52   51--0
	// sign   exponent   tail
	*/
	int maskBitCnt;
	long long inum = *((long long*)&fnum);
	inum = (inum >> 52) & 0x7FF;
	if (inum & 0x400)	// if the exponent of fnum is less than 1
		maskBitCnt = -inum & 0x3FF;
	maskBitCnt += precision;

	// mask bits from maskBitCnt to 0
	maskBitCnt = 51 - maskBitCnt;
	if (maskBitCnt >= 51)
		return 0;
	if (maskBitCnt < 0)
		return fnum;
	
	inum = *((long long*)&fnum);
	long long mask64 = 0x7FFFFFFF;
	mask64 >> 62 - maskBitCnt;
	inum &= mask64;
	return (double)inum;
}


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Utility
 *      Method:  combine
 * Description:  Combine two 32 bit integer and get a 64 bit integer.
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	int64	
Utility::combine64 (int int1, int int2)
{
	int64 combined = int1;
	combined = (combined << 32) + int2;
	return combined;
}		/* -----  end of method Utility::combine  ----- */


/*
 *--------------------------------------------------------------------------------------
 *        Class:  Utility
 *       Method:  Union
 *  Description:  Union two sets
 *        Param:
 *       Return:
 *--------------------------------------------------------------------------------------
 */
	void 
Utility::Union(vector<int>& sizes, vector<int>& leaders, int ele1, int ele2)
{
	int leader1 = SearchLeader(leaders, ele1);
	int leader2 = SearchLeader(leaders, ele2);

	if (leader1 == leader2) return;

	//cout << ele1 << "," << ele2 << "|" << leader1 << "," << leader2 << endl;
	if (sizes[leader1] < sizes[leader2])
	{
		leaders[leader1] = leader2;
		sizes[leader2] += sizes[leader1];
	}
	else
	{
		leaders[leader2] = leader1;
		sizes[leader1] += sizes[leader2];
	}
}		/* -----  end of method Utility::Union  ----- */


/*
 *--------------------------------------------------------------------------------------
 *        Class:  Utility
 *       Method:  SearchLeader
 *  Description:  Search the leader of a union tree
 *        Param:
 *       Return:
 *--------------------------------------------------------------------------------------
 */
	int 
Utility::SearchLeader(vector<int>& leaders, int element)
{
	int leader;
	int ele = element;

	while (leaders[ele] != ele)
		ele = leaders[ele];
	leader = ele;

	ele = element;
	while (leaders[ele] != ele)
	{
		int temp = ele;
		ele = leaders[ele];
		leaders[temp] = leader;
	}
	return leader;
}		/* -----  end of method Utility::SearchLeader  ----- */

/*
 *--------------------------------------------------------------------------------------
 *        Class:  Utility
 *       Method:  NextCombination
 *  Description:  Generate the next combination such that the numbers are in the range
 *                of [0, upper)
 *        Param:  Note, the last element of comb is a sentinel. Initialize the array
 *                like :
 *                for (int i = 0; i <= cnt; i++)
 *                    comb[i] = i;
 *                where cnt is the number of elements that are going to be selected.
 *       Return:  Whether there are new combinations
 *         Note:  The length of the array comb is at least len+1
 *--------------------------------------------------------------------------------------
 */
bool
Utility::NextCombination (int* comb, int len, int upper)
{
	// Find the right most index that could move right
	// The last element is a sentinel
	comb[len] = upper;
	int right_most = len - 1;
	while (right_most >= 0 && comb[right_most] + 1 == comb[right_most+1]) right_most--;

	if (right_most < 0) return false;

	// Move this right_most right for one step
	comb[right_most]++;
	// Move the next right one to right_most back to the adjacent position 
	// of right_most
	for (int i = right_most + 1; i < len; i++)
		comb[i] = comb[right_most]+i-right_most;
	
	return true;
}		/* -----  end of method Utility::NextCombination  ----- */

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInfer
 *       Method:  NextCombination
 *  Description:  Generate the next combination such that the numbers are in the range
 *                of [0, upper)
 *        Param:  Note, the last element of comb is a sentinel. Initialize the array
 *                like :
 *                for (int i = 0; i <= cnt; i++)
 *                    comb[i] = i;
 *                where cnt is the number of elements that are going to be selected.
 *       Return:  Whether there are new combinations
 *--------------------------------------------------------------------------------------
 */
bool
Utility::NextCombination (vector<int>& comb, int upper)
{
	// Find the right most index that could move right
	// The last element is a sentinel
	comb[comb.size()-1] = upper;
	int right_most = comb.size() - 2;
	while (right_most >= 0 && comb[right_most] + 1 == comb[right_most+1]) right_most--;

	if (right_most < 0) return false;

	// Move this right_most right for one step
	comb[right_most]++;
	// Move the next right one to right_most back to the adjacent position 
	// of right_most
	for (unsigned i = right_most + 1; i < comb.size(); i++)
		comb[i] = comb[right_most]+i-right_most;
	
	return true;
}		/* -----  end of method IsoInfer::NextCombination  ----- */

