/*
 * =====================================================================================
 *
 *       Filename:  Utility2.h
 *
 *    Description:  This file contains some common utilities
 *
 *        Version:  1.0
 *        Created:  04/09/2009 02:27:28 PM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef Utility2_H
#define Utility2_H

#include <vector>
#include <map>
#include <string>
#include "Utility.hpp"
#include <numeric>
#include <cmath>
#include <cassert>

// For Poisson distribution
#include <gsl/gsl_cdf.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

using namespace std;

typedef map<int64, double> map_64_double;
typedef map<int,int> map_int2int;
typedef map<int,string> map_int_str;

#define DEBUG

#ifndef DEBUG
	#define TRACE(_x)
#else
	#define TRACE(_x) cout << _x << endl;
#endif

#ifndef DEBUG
	#define LOCATION(_x)
#else
	#define LOCATION(_x) cout << "LOCATION: " << __FILE__ << ":" << __LINE__ << " " << _x << endl;
#endif

#ifndef DEBUG
	#define LOCATE
#else
	#define LOCATE cout << "LOCATION: " << __FILE__ << ":" << __LINE__ << endl;
#endif

#ifndef DEBUG
	#define ASSERT(_x, _y)
#else
	#define ASSERT(_x, _y) \
		if (!(_x)) cout << _y << endl; \
		assert(_x) 
#endif

class Utility2
{
	public:
	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  RandomNumInRange
	 *  Description:  Generate a random number in range
	 *        Param:  The random number will be in [low, high)
	 *                high > low and high - low < 65536;
	 *       Return:
	 *--------------------------------------------------------------------------------------
	 */
	static
	int
	RandomNumInRange(int low, int high)
	{
		if (high <= low)
			cerr << __func__ << " ERROR : high <= low" << endl;
		int64 ran = ((int64)rand()) * rand();
		return (int)(ran % (high - low) + low);
	}		/* -----  end of method Utility2::RandomNumInRange  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  RandomNumInRange
	 *  Description:  Select an index randomly with the probability proportional to given value
	 *        Param:  
	 *       Return:
	 *--------------------------------------------------------------------------------------
	 */
	static
	int
	RandomIndex(vector<double> prob)
	{
		vector<double> sum;
		sum.resize(prob.size());
		sum[0] = prob[0];
		for (unsigned i = 1; i < prob.size(); i++)
			sum[i] += sum[i-1] + prob[i];

		// Scale by rand_max
		double scale = (double) RAND_MAX / sum[sum.size()-1];
		for (unsigned i = 0; i < sum.size(); i++)
			sum[i] *= scale;
		sum[sum.size()-1] = RAND_MAX;

		int r = rand();
		unsigned idx = 0;

		while (r > sum[idx]) idx++;

		if (idx >= sum.size())
			cerr << "idx >= sum.size()" << endl;

		return idx;
	}		/* -----  end of method Utility2::RandomNumInRange  ----- */


	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  SumProbPoisson_Normalized
	 *  Description:  Let X_i~Poisson(lambda_i), Y_i=|X_i-E(X_i)|. Z=\sum_1^K Y_i. Given a value z_0, 
	 *                this function calculate the probability of observing Z>=z_0 by numeric method.
	 *        Param:  lambda  :  An array of lambda for each poisson distribution 
	 *        		  sample_cnt  :  The number of sample points used.
	 *       Return:
	 *--------------------------------------------------------------------------------------
	 */
	static
	double
	SumProbPoisson_Normalized(vector<double>& mu, double z0, int sample_cnt = 1000)
	{
		if (z0 == 0) return 1;

		int K = mu.size();
		const gsl_rng_type * T;
		gsl_rng * r;
		 
		gsl_rng_env_setup();

		T = gsl_rng_default;
		r = gsl_rng_alloc (T);
		 
		int cnt = 0;
		double all = 0;
		for (int i = 0; i < sample_cnt; i++)
		{
			double sum = 0;
			for (int j = 0; j < K; j++)
			{
				double lambda = mu[j];
				if (0 != lambda)
				{
				   double u = gsl_ran_poisson(r, lambda);
				   double scale = sqrt(lambda);
				   sum += std::abs(u - lambda) / scale;
				}
			}

			if (sum > z0)
			   cnt++;
			all += sum;
		}
		gsl_rng_free (r);

		return (double)cnt/sample_cnt;
	}		/* -----  end of method Utility2::SumProbPoisson_Normalized  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  ProbPoisson
	 *  Description:  Given 'lambda' of a poisson random variable, calculate the probability 
	 *                of observing 'error'
	 *        Param:  lambda  :  An array of lambda for each poisson distribution 
	 *        		  sample_cnt  :  The number of sample points used.
	 *       Return:
	 *--------------------------------------------------------------------------------------
	 */
	static
	double
	ErrorProbPoisson(double lambda, double error)
	{
		double prob = 0;
		if (lambda == 0 && error == 0)
			prob = 1;
		else if (lambda > 0)
			prob = gsl_cdf_poisson_P((int)(lambda - error), lambda) + 
				   gsl_cdf_poisson_Q((int)(lambda + error), lambda);
		if (prob > 1)
			prob = 1;
		return prob;
	}		/* -----  end of method Utility2::ErrorProbPoisson  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  NextCombination
	 *  Description:  Generate the next combination such that the numbers are in the range
	 *                of [0, upper)
	 *        Param:  Note, the last element of comb is a sentinel. Initialize the array
	 *                like :
	 *                for (int i = 0; i <= cnt; i++)
	 *                    comb[i] = i;
	 *                where cnt is the number of elements that are going to be selected.
	 *       Return:  Whether there are new combinations
	 *         Note:  The length of the array comb is at least len+1
	 *--------------------------------------------------------------------------------------
	 */
	static
	bool
	NextCombination (int* comb, int len, int upper)
	{
		// Find the right most index that could move right
		// The last element is a sentinel
		comb[len] = upper;
		int right_most = len - 1;
		while (right_most >= 0 && comb[right_most] + 1 == comb[right_most+1]) right_most--;

		if (right_most < 0) return false;

		// Move this right_most right for one step
		comb[right_most]++;
		// Move the next right one to right_most back to the adjacent position 
		// of right_most
		for (int i = right_most + 1; i < len; i++)
			comb[i] = comb[right_most]+i-right_most;
		
		return true;
	}		/* -----  end of method Utility2::NextCombination  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  IsoInfer
	 *       Method:  NextCombination
	 *  Description:  Generate the next combination such that the numbers are in the range
	 *                of [0, upper)
	 *        Param:  Note, the last element of comb is a sentinel. Initialize the array
	 *                like :
	 *                for (int i = 0; i <= cnt; i++)
	 *                    comb[i] = i;
	 *                where cnt is the number of elements that are going to be selected.
	 *       Return:  Whether there are new combinations
	 *--------------------------------------------------------------------------------------
	 */
	static
	bool
	NextCombination (vector<int>& comb, int upper)
	{
		// Find the right most index that could move right
		// The last element is a sentinel
		comb[comb.size()-1] = upper;
		int right_most = comb.size() - 2;
		while (right_most >= 0 && comb[right_most] + 1 == comb[right_most+1]) right_most--;

		if (right_most < 0) return false;

		// Move this right_most right for one step
		comb[right_most]++;
		// Move the next right one to right_most back to the adjacent position 
		// of right_most
		for (unsigned i = right_most + 1; i < comb.size(); i++)
			comb[i] = comb[right_most]+i-right_most;
		
		return true;
	}		/* -----  end of method IsoInfer::NextCombination  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  Union
	 *  Description:  Union two sets
	 *        Param:
	 *       Return:
	 *--------------------------------------------------------------------------------------
	 */
	static
	void
	Union(vector<int>& sizes, vector<int>& leaders, int ele1, int ele2)
	{
		int leader1 = SearchLeader(leaders, ele1);
		int leader2 = SearchLeader(leaders, ele2);

		if (leader1 == leader2) return;

		//cout << ele1 << "," << ele2 << "|" << leader1 << "," << leader2 << endl;
		if (sizes[leader1] < sizes[leader2])
		{
			leaders[leader1] = leader2;
			sizes[leader2] += sizes[leader1];
		}
		else
		{
			leaders[leader2] = leader1;
			sizes[leader1] += sizes[leader2];
		}
	}		/* -----  end of method Utility2::Union  ----- */


	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2
	 *       Method:  SearchLeader
	 *  Description:  Search the leader of a union tree
	 *        Param:
	 *       Return:
	 *--------------------------------------------------------------------------------------
	 */
	static
	int
	SearchLeader(vector<int>& leaders, int element)
	{
		int leader;
		int ele = element;

		while (leaders[ele] != ele)
			ele = leaders[ele];
		leader = ele;

		ele = element;
		while (leaders[ele] != ele)
		{
			int temp = ele;
			ele = leaders[ele];
			leaders[temp] = leader;
		}
		return leader;
	}		/* -----  end of method Utility2::SearchLeader  ----- */


};

template <typename TYPE = int> 
class Utility2Temp
{
	public:

	/*
	 *--------------------------------------------------------------------------------------
	 *        Class:  Utility2Temp
	 *       Method:  BinarySearch
	 *  Description:  Binary searching an integer in an array
	 *        Param:  data  :  An sorted (increasing) array
	 *                query :  The query 
	 *       Return:  return the index 'idx' such that idx is the smallest index which
	 *                satisfies data[idx] >= query.
	 *--------------------------------------------------------------------------------------
	 */
	static
	int
	BinarySearch(vector<TYPE>& data, TYPE query)
	{
		// binary search
		int low = 0;
		int high = data.size() - 1;
		int mid;

		while (low <= high)
		{
			mid = (low + high) / 2;

			if (data[mid] < query)
				low = mid + 1;
			else if (data[mid] > query)
				high = mid - 1;
			else if (data[mid] == query)
			{
				low = mid;
				break;
			}
		}
		return low;
	}		/* -----  end of method Utility2::BinarySearch ----- */
};

#endif
