// =====================================================================================
// 
//       Filename:  IsoInferPE.cc
// 
//    Description:  This is the implementation of class IsoInferPE
// 
//        Version:  1.0
//        Created:  06/08/2009 02:59:51 PM
//       Revision:  none
//       Compiler:  g++
// 
//         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
//        Company:  THU
// 
// =====================================================================================

#include <map>
#include <set>
#include <list>
#include <vector>
#include <queue>
#include <iostream>
#include <string>
#include <numeric>
#include <limits>
#include <algorithm>

// For sqrt()
#include <cmath>

// For rand()
#include <time.h>

// For setw()
#include <iomanip>

// For split
#include "Utility.hpp"
#include "Utility2.h"

#include "IsoInferPE.h"
#include "InstanceWriter.h"
#include "GraphAlgorithm_Basic.hpp"
#include "glpk.h"
#include "GraphWriter.h"

IsoInferPE::IsoInferPE (LPsolver* p_solver, ostream* p_output) : InstanceHandler(p_output)
{
	mpSolver = p_solver;

	mbEnableStartEnd = true;
	mbEnablePE = true;     
	mbEnableStepII = true;
	mMinEffectivePartCombDup = 1;
	mMinExpLevel = 5;
	mOmittedJunctionCnt = 0;
	mConfidenceLevel = 0.05;

	mPartitionSize = 10;
} 							   /* constructor      */

IsoInferPE::~IsoInferPE ()
{
};                            /* destructor       */

/*virtual*/
void
IsoInferPE::Initialize()
{
}

/*virtual*/
void
IsoInferPE::OnInstance(Instance& an_instance)
{
	mpInstance = &an_instance;
	mMaxValidIsoformCnt = 1000;

	vector<vector<int> > start_exons = an_instance.mStartExons;
	vector<vector<int> > end_exons = an_instance.mEndExons;
	vector<vector<double> > junc_cnt =  an_instance.mSpliceReadCnt;
	vector<vector<bool> > isoforms =  an_instance.mIsoforms;
	vector<PEInfo>& pe_info =  an_instance.mPEInfo;
	vector<Exon>& exons = an_instance.mExons;

	vector<int> set_sizes;
	vector<double> start_from_cnt;         // The number of reads starting from an exon
	vector<double> end_at_cnt;             // The number of reads starting from an exon
	set_sizes.resize(exons.size());
	start_from_cnt.resize(set_sizes.size());
	end_at_cnt.resize(set_sizes.size());
	for (unsigned i = 0; i < start_from_cnt.size(); i++)
	{
		set_sizes[i] = exons[i].mEnd - exons[i].mStart;
		start_from_cnt[i] = exons[i].mStartCnt;
		end_at_cnt[i] = exons[i].mEndCnt;
	}

	mValidIsoforms.clear();
	mSolution.clear();

	// Always connect two consecutive segments of an exon if the
	// boundary has the same type. However, this step may introduce false
	// positives if an segment should be intron but falsely kepted as an
	// expressed segments
	for (unsigned i = 0; i < junc_cnt.size()-1; i++)
	{
		// In this case exons[i].mEndType == exons[i+1].mStartType)
		if (junc_cnt[i][i+1] == 0 && exons[i].mEnd == exons[i+1].mStart &&
			exons[i].mEndType != 2)
			junc_cnt[i][i+1] = 1;
	}


	if (!mbEnableStartEnd)
	{
		start_exons.clear();
		end_exons.clear();

		// Only segments with 0 in degree can be the start segments
		// Only segments with 0 out degree can be the end segments
		vector<int> a_vec;
		for (unsigned i = 0; i < junc_cnt.size(); i++)
		{
			bool b_succ = true;
			for (unsigned j = 0; j < i; j++)
				if (junc_cnt[j][i] > 0)
				{
					b_succ = false;
					break;
				}
			if (b_succ)
				a_vec.push_back(i);
		}
		start_exons.push_back(a_vec);
		a_vec.clear();

		for (unsigned i = 0; i < junc_cnt.size(); i++)
		{
			bool b_succ = true;
			for (unsigned j = i+1; j < junc_cnt.size(); j++)
				if (junc_cnt[i][j] > 0)
				{
					b_succ = false;
					break;
				}
			if (b_succ)
				a_vec.push_back(i);
		}
		end_exons.push_back(a_vec);
	}

	vector<vector<bool> >& known_isoforms = mKnownIsoforms;

	GraphEx<int> valid_graph;
	int source;
	int sink;
	BuildSpliceGraph(valid_graph, junc_cnt, start_exons, end_exons, source, sink);

	list<set<int> > components;
	vector<int> source_sink;
	source_sink.push_back(source);
	source_sink.push_back(sink);

	valid_graph.MaskNode(source_sink, true);
	valid_graph.SetDirected(false);
	GraphAlgorithm_Basic::Components(&valid_graph, components);

	cout << "It has been decomposed into " << components.size() << " subinstances" << endl;

	// Each connected component corresponds to a subinstance
	for_each_ele_in_group(iter, list<set<int> >, components)
	{
		vector<int> sub_set;
		for_each_ele_in_group(iter2, set<int>, *iter)
		{
			int in_id = *iter2;
			if (in_id != source && in_id != sink)
			{
				int ex_id = valid_graph.GetNodeExID(in_id);
				sub_set.push_back(ex_id);
			}
		}
		sort(sub_set.begin(), sub_set.end());

		cout << "Sub instance : ";
		for (unsigned i = 0; i < sub_set.size(); i++)
			cout << sub_set[i] << "\t";
		cout << endl;

		vector<int> map_to_flat_idx;
		map_to_flat_idx.assign(set_sizes.size(), -1);
		for (unsigned i = 0; i < sub_set.size(); i++)
			map_to_flat_idx[sub_set[i]] = i;

		vector<vector<int> > start_exons_pro;
		vector<vector<int> > end_exons_pro;
		for (unsigned i = 0; i < start_exons.size(); i++)
		{
			vector<int> starts;
			for (unsigned j = 0; j < start_exons[i].size(); j++)
				if (-1 != map_to_flat_idx[start_exons[i][j]])
					starts.push_back(map_to_flat_idx[start_exons[i][j]]);

			vector<int> ends;
			for (unsigned j = 0; j < end_exons[i].size(); j++)
				if (-1 != map_to_flat_idx[end_exons[i][j]])
					ends.push_back(map_to_flat_idx[end_exons[i][j]]);

			if (starts.size() == 0 || ends.size() == 0) continue;

			bool b_exist = false;
			for (unsigned j = 0; j < start_exons_pro.size(); j++)
			{
				if (starts == start_exons_pro[j] && ends == end_exons_pro[j])
				{
					b_exist = true;
					break;
				}
			}

			if (!b_exist)
			{
				start_exons_pro.push_back(starts);
				end_exons_pro.push_back(ends);
			}
		}

		if (start_exons_pro.size() == 0) continue;

		vector<int> set_sizes_pro;
		vector<double> start_from_cnt_pro;
		vector<double> end_at_cnt_pro;
		vector<vector<double> > junc_cnt_pro;
		vector<PEInfo> pe_info_pro;
		vector<int> exon_type;
		vector<int> exon_type_pro;
		exon_type.assign(set_sizes.size(), 0);

		Project(set_sizes, set_sizes_pro,
				exon_type, exon_type_pro,
				start_from_cnt, start_from_cnt_pro,
				end_at_cnt, end_at_cnt_pro,
				junc_cnt, junc_cnt_pro,
				pe_info, pe_info_pro, sub_set);

		vector<vector<bool> > known_isoforms_pro;
		vector<vector<int> > known_isoforms_group;
		Project(known_isoforms, known_isoforms_pro, sub_set, known_isoforms_group);

		vector<vector<bool> > valid_isoforms;
		vector<int> solution;
		OnSubInstance (set_sizes_pro, start_from_cnt_pro, end_at_cnt_pro,
					   junc_cnt_pro, start_exons_pro, end_exons_pro,
					   pe_info_pro, known_isoforms_pro, valid_isoforms, solution);

		int old_size = mValidIsoforms.size();

		for (unsigned i = 0; i < valid_isoforms.size(); i++)
		{
			vector<bool> an_iso;
			an_iso.assign(set_sizes.size(), false);
			for (unsigned j = 0; j < valid_isoforms[i].size(); j++)
				an_iso[sub_set[j]] = valid_isoforms[i][j];
			mValidIsoforms.push_back(an_iso);
		}

		for (unsigned i = 0; i < solution.size(); i++)
			mSolution.push_back(solution[i] + old_size);
	}
	
	InstanceHandler::OnInstance(an_instance);
}

void
IsoInferPE::OnSubInstance ( const vector<int>& set_sizes,
							const vector<double>& start_from_cnt,
							const vector<double>& end_at_cnt,
							const vector<vector<double> >& junc_cnt, 
							const vector<vector<int> >& start_exons, 
							const vector<vector<int> >& end_exons,
							const vector<PEInfo>& pe_info, 
							const vector<vector<bool> >& known_isoforms,
							vector<vector<bool> >& valid_isoforms_expanded,
							vector<int>& solution)
{
	vector<int> shrink_map1;
	vector<int> set_sizes_shrinked1;
	vector<double> start_from_cnt_shrinked1;
	vector<double> end_at_cnt_shrinked1;
	vector<vector<double> > junc_cnt_shrinked1;
	vector<PEInfo> pe_info_shrinked1;
	vector<vector<int> > start_exons_shrinked1;
	vector<vector<int> > end_exons_shrinked1;

	vector<vector<bool> > valid_isoforms;
	vector<int> valid_isoform_order;


	CalculateShrink(junc_cnt, start_exons, end_exons, shrink_map1);
	
	ShrinkInstance(shrink_map1, 
					set_sizes, set_sizes_shrinked1, 
					start_from_cnt, start_from_cnt_shrinked1, 
					end_at_cnt, end_at_cnt_shrinked1, 
					junc_cnt, junc_cnt_shrinked1,
					pe_info, pe_info_shrinked1,
					start_exons, start_exons_shrinked1, 
					end_exons, end_exons_shrinked1);

	vector<vector<bool> > known_isoforms_shrinked1;
	ShrinkIsoform(shrink_map1, known_isoforms, known_isoforms_shrinked1);

	EnumerateValidByExpLevel(junc_cnt_shrinked1, start_exons_shrinked1, end_exons_shrinked1,
					set_sizes_shrinked1, pe_info_shrinked1, valid_isoforms, valid_isoform_order);

//	for (unsigned i = 0; i < valid_isoforms.size(); i++)
//	{
//		for (unsigned j = 0; j < valid_isoforms[i].size(); j++)
//			cout << valid_isoforms[i][j] << " ";
//		cout << endl;
//	}

	//Remove valid_isoforms that are known
	int cnt = 0;
	for (unsigned i = 0; i < valid_isoforms.size(); i++)
	{
		bool b_known = false;
		for (unsigned j = 0; j < known_isoforms_shrinked1.size(); j++)
			if (valid_isoforms[i] == known_isoforms_shrinked1[j])
			{
				b_known = true;
				break;
			}

		if (!b_known)
			valid_isoforms[cnt++] = valid_isoforms[i];
	}
	valid_isoforms.resize(cnt);

	mWeight.assign(valid_isoforms.size(), 1);
	if (mbEnablePE)
	{
		for (unsigned i = 0; i < valid_isoforms.size(); i++)
		{
			double max_exp_in_rpkm = MaxConsistExp(valid_isoforms[i], set_sizes_shrinked1, pe_info_shrinked1);
			mWeight[i] = max_exp_in_rpkm;
		}

		vector<int> sortedIdx;
		UtilityTemp<double>::Sort(mWeight, sortedIdx);
		UtilityTemp<vector<bool> >::SortByIndex(valid_isoforms, sortedIdx);

		double order = 1;
		for (unsigned i = 0; i < mWeight.size(); i++)
		{
			if (i > 0 && mWeight[i] != mWeight[i-1])
				order += 1;
			mWeight[i] = order;
		}
	}

	// Append known_isoforms to the valid_isoforms
	vector<bool> b_known_isoforms;
	b_known_isoforms.assign(valid_isoforms.size(), false);
	for (unsigned i = 0; i < known_isoforms_shrinked1.size(); i++)
	{
		valid_isoforms.push_back(known_isoforms_shrinked1[i]);
		b_known_isoforms.push_back(true);
		mWeight.push_back(0);
	}

	vector<int> shrink_map2;
	vector<int> set_sizes_shrinked2;
	vector<double> start_from_cnt_shrinked2;
	vector<double> end_at_cnt_shrinked2;
	vector<vector<bool> > valid_isoforms_shrinked2;
	vector<vector<double> > junc_cnt_shrinked2;
	vector<PEInfo> pe_info_shrinked2;
	vector<vector<int> > start_exons_shrinked2;
	vector<vector<int> > end_exons_shrinked2;
	vector<double> exp_shrinked2;

	if (valid_isoforms.size() != 0)
	{

//		shrink_map2.resize(valid_isoforms[0].size());
//		for (unsigned i = 0; i < shrink_map2.size(); i++)
//			shrink_map2[i] = i;

		// Shrink the instance according to valid isoforms.
		CalculateShrink(valid_isoforms, shrink_map2);

		ShrinkInstance(shrink_map2,
					set_sizes_shrinked1, set_sizes_shrinked2, 
					start_from_cnt_shrinked1, start_from_cnt_shrinked2, 
					end_at_cnt_shrinked1, end_at_cnt_shrinked2, 
					junc_cnt_shrinked1, junc_cnt_shrinked2,
					pe_info_shrinked1, pe_info_shrinked2,
					start_exons_shrinked1, start_exons_shrinked2, 
					end_exons_shrinked1, end_exons_shrinked2);

		ShrinkIsoform(shrink_map2, valid_isoforms, valid_isoforms_shrinked2);

		// It is possible that some start-end exon pair does not appear in valid_isoforms.
		// Therefore, reextract start-end exon pair from shrinked valid isoforms.

		cout << "ShrinkInstance : from " << set_sizes.size() << " to " << shrink_map1.size()
			<< " to " << shrink_map2.size() << " with " << valid_isoforms_shrinked2.size() 
			<< " valid isoforms." << endl;

		// Merge shrink_map1 and shrink_map2
		for (unsigned i = 0; i < shrink_map1.size(); i++)
			shrink_map1[i] = shrink_map2[shrink_map1[i]];

		/*
		// Debug
		cout << "Map1 : ";
		for (unsigned i = 0; i < shrink_map1.size(); i++)
			cout << shrink_map1[i] << "\t";
		cout << endl;
		cout << "Map2 : ";
		for (unsigned i = 0; i < shrink_map2.size(); i++)
			cout << shrink_map2[i] << "\t";
		cout << endl;
		for (unsigned i = 0; i < valid_isoforms.size(); i++)
		{
			for (unsigned j = 0; j < valid_isoforms[i].size(); j++)
				cout << valid_isoforms[i][j] << " ";
			cout << endl;
		}
		for (unsigned i = 0; i < pe_info_shrinked2[0].mPartComb.size(); i++)
		{
			for (unsigned j = 0; j < pe_info_shrinked2[0].mPartComb[i].size(); j++)
				cout << pe_info_shrinked2[0].mPartComb[i][j] << " ";
			cout << pe_info_shrinked2[0].mPartCombDup[i] << endl;
		}
		  */

		if (mbEnableStepII)
			InferNew(set_sizes_shrinked2, start_from_cnt_shrinked2, end_at_cnt_shrinked2, junc_cnt_shrinked2, 
					valid_isoforms_shrinked2, b_known_isoforms, start_exons_shrinked2, end_exons_shrinked2, pe_info_shrinked2, 
					solution);

		ExpandIsoforms(shrink_map1, valid_isoforms_shrinked2, valid_isoforms_expanded);
	}
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  CalculateShrink
// Description:  Given junction information, calculate the shrinking
//  Parameters:  shrink_map[i] stores the exon index in the new instance 
//               corresponding to i'th exon in the original instance. 
//        Note:  The shrinking will be consistent with known_isoforms
//      Return:  The size after shrinking
//--------------------------------------------------------------------------------------
int 
IsoInferPE::CalculateShrink ( const vector<vector<double> >& junc_cnt, const vector<vector<int> >& start_exons, 
							  const vector<vector<int> >& end_exons, vector<int>& shrink_map)
{
	GraphEx<int> valid_graph;
	int source;
	int sink;
	BuildSpliceGraph(valid_graph, junc_cnt, start_exons, end_exons, source, sink);

	int largest_idx = 0;
	shrink_map.resize(junc_cnt.size());
	for (unsigned i = 0; i < junc_cnt.size(); i++)
	{
		shrink_map[i] = largest_idx;
		if (i > 0)
		{
			int curr_node = valid_graph.GetNodeInID(i);
			const int* in_edges = valid_graph.InEdges(curr_node);
			if (valid_graph.InDegree(curr_node) == 1)
			{
				int prev_node = valid_graph.FromNode(in_edges[0]);
				if (valid_graph.OutDegree(prev_node) == 1)
				{
					int exon_id = valid_graph.GetNodeExID(prev_node);
					// if (!b_keep_order || exon_id == i-1) 
					// Note that the order don't have to be kept.
					if (exon_id >= 0 && exon_id < junc_cnt.size())
						shrink_map[i] = shrink_map[exon_id];
				}
			}
		}
		if (shrink_map[i] == largest_idx) largest_idx++;
	}
	return largest_idx;
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  CalculateShrink
// Description:  Given a set of isoforms, calculate the shrinking
//  Parameters:  shrink_map[i] stores the exon index in the new instance 
//               corresponding to i'th exon in the original instance. 
//        Note:  
//      Return:  The size after shrinking
//--------------------------------------------------------------------------------------
int
IsoInferPE::CalculateShrink ( const vector<vector<bool> >& isoforms, vector<int>& shrink_map)
{
	ASSERT((isoforms.size() > 0), "There are 0 isoforms used to calculate shrinking");

	vector<vector<bool> > tran_isoforms;
	tran_isoforms.resize(isoforms[0].size());
	for (unsigned i = 0; i < tran_isoforms.size(); i++)
	{
		tran_isoforms[i].resize(isoforms.size());
		for (unsigned j = 0; j < isoforms.size(); j++)
			tran_isoforms[i][j] = isoforms[j][i];
	}

	int set_cnt = isoforms[0].size();

	shrink_map.resize(set_cnt);
	int exon_idx = 0;
	// Only shrink consecutive exons.
	for (int i = 0; i < set_cnt; i++)
	{
		if (i > 0 && tran_isoforms[i-1] != tran_isoforms[i])
			exon_idx++;
		shrink_map[i] = exon_idx;
	}
	return exon_idx+1;
}


//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ShrinkInstance
// Description:  If a set of consecutive exons always appear or disappear together
//				 in the given isoforms, they can be treated as a single "exon". This should 
//				 improve the result and help to handle instances with a large number 
//				 of exons. 
//  Parameters:  ...   :  The original instance
//               ..._shrinked :  The shrinked instance
//               isoforms  : Normally, pass all valid_isoforms to this parameter. It will
//                   be used to determine which exons should be grouped together.
//        Note:  start_exons_shrinked and end_exons_shrinked are construct from 
//               valid_isoforms_shrinked. Therefore, the original start_exons and end_exons
//               are not needed.
//--------------------------------------------------------------------------------------
	void
IsoInferPE::ShrinkInstance (const vector<int>& shrink_map,
							const vector<int>& set_sizes, vector<int>& set_sizes_shrinked, 
							const vector<double>& start_from_cnt, vector<double>& start_from_cnt_shrinked, 
							const vector<double>& end_at_cnt, vector<double>& end_at_cnt_shrinked, 
							const vector<vector<double> >& junc_cnt, vector<vector<double> >& junc_cnt_shrinked,
							const vector<PEInfo>& pe_info, vector<PEInfo>& pe_info_shrinked,
							const vector<vector<int> >& start_exons, vector<vector<int> >& start_exons_shrinked, 
							const vector<vector<int> >& end_exons, vector<vector<int> >& end_exons_shrinked)
{
	vector<vector<int> > grouped_exons;
	for (unsigned i = 0; i < shrink_map.size(); i++)
	{
		if (grouped_exons.size() <= shrink_map[i]) 
			grouped_exons.resize(shrink_map[i]+1);
		grouped_exons[shrink_map[i]].push_back(i);
	}

	set_sizes_shrinked.resize(grouped_exons.size());
	start_from_cnt_shrinked.resize(grouped_exons.size());
	end_at_cnt_shrinked.resize(grouped_exons.size());
	for (unsigned i = 0; i < grouped_exons.size(); i++)
	{
		//DEBUG
		assert(grouped_exons[i].size() > 0);

		double from_read_cnt = 0;
		double to_read_cnt = 0;
		int exon_len = 0;
		for (int j = 0; j < grouped_exons[i].size(); j++)
		{
			int idx = grouped_exons[i][j];
			if (set_sizes[idx] > 0)
			{
				exon_len += set_sizes[idx];
				from_read_cnt += start_from_cnt[idx];
				to_read_cnt += end_at_cnt[idx];
			}
		}

		set_sizes_shrinked[i] = exon_len;
		start_from_cnt_shrinked[i] = from_read_cnt;
		end_at_cnt_shrinked[i] = to_read_cnt;
	}

	// for junc_cnt_shrinked
	// Note that the junctions omitted in EnumerateValid will not appear in the
	// shrinked junctions.
	junc_cnt_shrinked.resize(grouped_exons.size());
	for (unsigned i = 0; i < junc_cnt_shrinked.size(); i++)
	{
		junc_cnt_shrinked[i].assign(junc_cnt_shrinked.size(), 0);
		for (unsigned j = i + 1; j < junc_cnt_shrinked.size(); j++)
		{
			int from;
			int to;
			if (grouped_exons[i][grouped_exons[i].size()-1] < grouped_exons[j][0])
			{
				from = grouped_exons[i][grouped_exons[i].size()-1];
				to = grouped_exons[j][0];
				junc_cnt_shrinked[i][j] = junc_cnt[from][to];
			}else if (grouped_exons[j][grouped_exons[j].size()-1] < grouped_exons[i][0])
			{
				from = grouped_exons[j][grouped_exons[j].size()-1];
				to = grouped_exons[i][0];
				junc_cnt_shrinked[i][j] = junc_cnt[from][to];
			}
			// Otherwise, no junction reads should be observed between the two group of exons.
		}
	}

	pe_info_shrinked.resize(pe_info.size());
	for (unsigned k = 0; k < pe_info.size(); k++)
	{
		PEInfo& curr_info_shrinked = pe_info_shrinked[k];

		curr_info_shrinked.mSpanLow = pe_info[k].mSpanLow;
		curr_info_shrinked.mSpanHigh = pe_info[k].mSpanHigh;
		curr_info_shrinked.mSpanMean = pe_info[k].mSpanMean;
		curr_info_shrinked.mSpanStd = pe_info[k].mSpanStd;
		curr_info_shrinked.mReadLen = pe_info[k].mReadLen;
		curr_info_shrinked.mReadCnt = pe_info[k].mReadCnt;

		const vector<vector<bool> >& old_part_comb = pe_info[k].mPartComb;

		vector<vector<bool> > temp_part_comb;
		vector<int> temp_part_comb_dup = pe_info[k].mPartCombDup;

		// for old_part_comb
		for (unsigned i = 0; i < old_part_comb.size(); i++)
		{
			vector<bool> a_part_comb;
			a_part_comb.assign(grouped_exons.size(), false);
			for (unsigned j = 0; j < old_part_comb[i].size(); j++)
			{
				int idx = shrink_map[j];
				a_part_comb[idx] = a_part_comb[idx] | old_part_comb[i][j];
			}
			temp_part_comb.push_back(a_part_comb);
		}

		// remove duplications in temp_part_comb and combine read cnt
		vector<int> group_part_comb;
		group_part_comb.resize(temp_part_comb.size());
		for (unsigned i = 0; i < group_part_comb.size(); i++)
			group_part_comb[i] = i;

		for (unsigned i = 0; i < temp_part_comb.size(); i++)
			for (unsigned j = i+1; j < temp_part_comb.size(); j++)
				if (temp_part_comb[i] == temp_part_comb[j])
					Utility2::Union(temp_part_comb_dup, group_part_comb, i, j);

		for (unsigned i = 0; i < temp_part_comb.size(); i++)
			Utility2::SearchLeader(group_part_comb, i);

		vector<bool> mask;
		mask.assign(temp_part_comb.size(), false);
		for (unsigned i = 0; i < temp_part_comb.size(); i++)
			mask[group_part_comb[i]] = true;

		vector<vector<bool> >& part_comb_shrinked = curr_info_shrinked.mPartComb;
		vector<int>& part_comb_dup_shrinked = curr_info_shrinked.mPartCombDup;

		part_comb_shrinked.clear();
		part_comb_dup_shrinked.clear();
		for (unsigned i = 0; i < mask.size(); i++)
			if (mask[i])
			{
				part_comb_shrinked.push_back(temp_part_comb[i]);
				part_comb_dup_shrinked.push_back(temp_part_comb_dup[i]);
			}
	}

	// For start_exons_shrinked and end_exons_shrinked
	start_exons_shrinked.resize(start_exons.size());
	end_exons_shrinked.resize(end_exons.size());

	for (unsigned i = 0; i < start_exons.size(); i++)
	{
		set<int> indexes;
		for (unsigned j = 0; j < start_exons[i].size(); j++)
			indexes.insert(shrink_map[start_exons[i][j]]);
		start_exons_shrinked[i].assign(indexes.begin(), indexes.end());

		indexes.clear();
		for (unsigned j = 0; j < end_exons[i].size(); j++)
			indexes.insert(shrink_map[end_exons[i][j]]);
		end_exons_shrinked[i].assign(indexes.begin(), indexes.end());

		// Do current start and end exons appear before?
		for (unsigned j = 0; j < i; j++)
		{
			if (start_exons_shrinked[j] == start_exons_shrinked[i] &&
			    end_exons_shrinked[j] == end_exons_shrinked[i])
			{
				set<int> uniq;
				for (unsigned k = 0; k < start_exons_shrinked[j].size(); k++)
					uniq.insert(start_exons_shrinked[j][k]);
				for (unsigned k = 0; k < start_exons_shrinked[i].size(); k++)
					uniq.insert(start_exons_shrinked[i][k]);
				start_exons_shrinked[j].assign(uniq.begin(), uniq.end());

				uniq.clear();
				for (unsigned k = 0; k < end_exons_shrinked[j].size(); k++)
					uniq.insert(end_exons_shrinked[j][k]);
				for (unsigned k = 0; k < end_exons_shrinked[i].size(); k++)
					uniq.insert(end_exons_shrinked[i][k]);
				end_exons_shrinked[j].assign(uniq.begin(), uniq.end());

				start_exons_shrinked[i].clear();
				end_exons_shrinked[i].clear();
			}
		}
	}

	int cnt = 0;
	for (unsigned i = 0; i < start_exons_shrinked.size(); i++)
	{
		if (start_exons_shrinked[i].size() != 0 && end_exons_shrinked[i].size() != 0)
		{
			start_exons_shrinked[cnt] = start_exons_shrinked[i];
			end_exons_shrinked[cnt] = end_exons_shrinked[i];
			cnt++;
		}
	}

	return ;
}		// -----  end of method IsoInferPE::ShrinkInstance  -----

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ExtractStartEndExons
// Description:  Extract the start-end exon pair of each isoform
//  Parameters:  
//--------------------------------------------------------------------------------------
	void
IsoInferPE::ExtractStartEndExons(const vector<vector<bool> >& isoforms, vector<int>& start_exons,
								 vector<int>& end_exons)
{
	int exon_cnt = isoforms[0].size();
	start_exons.resize(isoforms.size());
	end_exons.resize(isoforms.size());
	for (unsigned iso_cnt = 0; iso_cnt < isoforms.size(); iso_cnt++)
	{
		int start = 0;
		int end = exon_cnt - 1;
		for (int start_exon = 0; start_exon < exon_cnt; start_exon++)
			if (isoforms[iso_cnt][start_exon])
			{
				start = start_exon;
				break;
			}

		for (int end_exon = exon_cnt - 1; end_exon >= 0; end_exon--)
			if (isoforms[iso_cnt][end_exon])
			{
				end = end_exon;
				break;
			}
		ASSERT((start < exon_cnt && end >= 0), "An empty isoform is encountered");
		start_exons[iso_cnt] = start;
		end_exons[iso_cnt] = end;
	}
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ShrinkIsoform
// Description:  Call this function after ShrinkInstance is called. This function 
//               shrink the given isoforms.
//  Parameters:  
//--------------------------------------------------------------------------------------
	void
IsoInferPE::ShrinkIsoform( const vector<int>& shrink_map, const vector<vector<bool> >& isoforms, 
							vector<vector<bool> >& isoforms_shrinked)
{
	int new_exon_cnt = 0;
	for (unsigned i = 0; i < shrink_map.size(); i++)
		if (new_exon_cnt < shrink_map[i]) new_exon_cnt = shrink_map[i];
	new_exon_cnt++;

	isoforms_shrinked.clear();
	for (unsigned i = 0; i < isoforms.size(); i++)
	{
		// Check whether the shrinking is consistent with the existing isoforms 
		bool succ = true;
		vector<bool> b_visited;
		vector<bool> an_iso;
		b_visited.assign(new_exon_cnt, false);
		an_iso.assign(new_exon_cnt, false);

		for (unsigned j = 0; j < shrink_map.size(); j++)
		{
			int idx = shrink_map[j];
			if (b_visited[idx] && an_iso[idx] != isoforms[i][j])
			{
				succ = false; break;
			}
			else
				an_iso[idx] = isoforms[i][j];
		}

		if (!succ)
		{
			cerr << "WARNING : isoform " << i << " is inconsistent with the shrinking. This isoform is skipped." << endl;
			continue;
		}

		isoforms_shrinked.push_back(an_iso);
	}
	return ;
}		// -----  end of method IsoInferPE::ShrinkIsoform  -----

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ExpandIsoforms
// Description:  Given a set of isoforms based on the shrinked instance, this method 
//               expands it to the isoforms for the original problem.
//  Parameters:
//--------------------------------------------------------------------------------------
	void
IsoInferPE::ExpandIsoforms ( const vector<int>& shrink_map, const vector<vector<bool> >& isoforms_shrinked, 
							vector<vector<bool> >& isoforms_expanded )
{
	isoforms_expanded.resize(isoforms_shrinked.size());
	for (unsigned i = 0; i < isoforms_shrinked.size(); i++)
	{
		isoforms_expanded[i].resize(shrink_map.size());
		for (int j = 0; j < shrink_map.size(); j++)
			isoforms_expanded[i][j] = isoforms_shrinked[i][shrink_map[j]];
	}

	return ;
}		// -----  end of method IsoInferPE::ExpandIsoforms  -----



//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  BuildSpliceGraph
// Description:  Build the splice graph based on splice reads information and the start
//               and end exon information
//  Parameters:  start_exons  :  All possible start exons. If the length of start_exons 
//                   is zero, all the exons can be the start exon.
//               end_exons    :  All possible end exons. If the length of end_exons is
//                   zeror, all the exons can be the end exon.
//               source[o]    :  The internal ID of the source in the graph
//               sink[o]      :  The internal ID of the sink in the graph
//--------------------------------------------------------------------------------------
	void
IsoInferPE::BuildSpliceGraph ( GraphEx<int>& splice_graph, const vector<vector<double> >& junc_cnt, 
		const vector<vector<int> >& start_exons, const vector<vector<int> >& end_exons, int& source, int& sink)
{
	int set_cnt = junc_cnt[0].size();

	splice_graph.BeginAddNodeOrEdge();
	for (int i = 0; i < set_cnt; i++)
		splice_graph.AddNodeEx(i);

	source = -1;
	sink = -2;

	splice_graph.AddNodeEx(source); // source;
	splice_graph.AddNodeEx(sink); // sink;

	// Add edges according to splice information.
	for (int i = 0; i < set_cnt; i++)
		for (int j = i+1; j < set_cnt; j++)
			if (junc_cnt[i][j] >= mMinEffectivePartCombDup) splice_graph.AddEdgeEx(i, j, i * set_cnt + j);

	set<int> flat_start_exons;
	set<int> flat_end_exons;
	for (unsigned i = 0; i < start_exons.size(); i++)
	{
		for (unsigned j = 0; j < start_exons[i].size(); j++)
			flat_start_exons.insert(start_exons[i][j]);
		for (unsigned j = 0; j < end_exons[i].size(); j++)
			flat_end_exons.insert(end_exons[i][j]);
	}

	int ex_edge_id = set_cnt*set_cnt;

	// Add edges from the source to each exon that can be the first exon
	for_each_ele_in_group(iter, set<int>, flat_start_exons)
		splice_graph.AddEdgeEx(source, *iter, ex_edge_id++);
	// Add edges from each exon that can be the end exon to the sink 
	for_each_ele_in_group(iter, set<int>, flat_end_exons)
		splice_graph.AddEdgeEx(*iter, sink, ex_edge_id++);

	splice_graph.EndAddNodeOrEdge();

	source = splice_graph.GetNodeInID(source);
	sink = splice_graph.GetNodeInID(sink);

	return ;
}		// -----  end of method IsoInferPE::BuildSpliceGraph  -----

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ConstructLPInstance
// Description:  Construct a LP instance
//               
//  Parameters:  A measure could be an exon, junction, pair or triple
//--------------------------------------------------------------------------------------
void
IsoInferPE::ConstructLPInstance(const vector<int>& set_sizes, 
								const vector<int>& exon_type,
								const vector<double>& start_from_cnt, 
								const vector<double>& end_at_cnt, 
								const vector<vector<double> >& junc_cnt, 
								const vector<vector<bool> >& isoforms,
								const vector<PEInfo>& pe_info,
								vector<vector<int> >& measure_in_isoform,
								vector<vector<double> >& measure_virtual_length,
								vector<double>& measure_read)
{
	// Calculate the virtual lengths of exons and junctions.
	ExonJuncVirtualLength(set_sizes, exon_type, start_from_cnt, end_at_cnt, isoforms, junc_cnt, 
							measure_in_isoform, measure_read, measure_virtual_length);
	if (measure_read.size() == 0) return;

	int junc_len = mpInstance->mReadLen - mpInstance->mCrossStrength + 1;
	// virtual length * expression level (in RPKM) = #observed reads
	double ratio = ((double)mpInstance->mReadCnt) / 1000000000.0;
	double junc_v_len = junc_len * ratio;

	// Remove exons or junctions with too small virtual lengths
	// Remove measures with zero reads
	vector<bool> b_measure_removed;
	b_measure_removed.assign(measure_read.size(), false);

	// Remove any measure that some isoform has non-positive virtual length
	for (unsigned i = 0; i < measure_virtual_length.size(); i++)
		for (unsigned j = 0; j < measure_virtual_length[i].size(); j++)
			if (measure_virtual_length[i][j] <= 0 ) b_measure_removed[measure_in_isoform[i][j]] = true;

	RemoveMeasures(measure_in_isoform, measure_virtual_length, measure_read, b_measure_removed);
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  CombineMeasure
// Description:  Combine the second set of measures into the first set of measures.
//               
//  Parameters:  
//--------------------------------------------------------------------------------------
void
IsoInferPE::CombineMeasure(vector<vector<int> >& first_measure_in_isoform,
							vector<vector<double> >& first_measure_virtual_length,
							vector<double>& first_measure_read,
							const vector<vector<int> >& second_measure_in_isoform,
							const vector<vector<double> >& second_measure_virtual_length,
							const vector<double>& second_measure_read)
{
	unsigned base = first_measure_read.size();
	for (unsigned i = 0; i < second_measure_in_isoform.size(); i++)
	{
		for (unsigned j = 0; j < second_measure_in_isoform[i].size(); j++)
		{
			first_measure_in_isoform[i].push_back(base + second_measure_in_isoform[i][j]);
			first_measure_virtual_length[i].push_back(base + second_measure_virtual_length[i][j]);
		}
	}

	for (unsigned i = 0; i < second_measure_read.size(); i++)
		first_measure_read.push_back(second_measure_read[i]);
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  LPInstance
// Description:  Construct a LP instance
//               
//  Parameters:  A measure could be an exon, junction, pair or triple
//--------------------------------------------------------------------------------------
void
IsoInferPE::RemoveMeasures (vector<vector<int> >& measure_in_isoform,
							vector<vector<double> >& measure_virtual_length,
							vector<double>& measure_read,
							const vector<bool>& b_measure_removed)
{
	vector<int> map_to_flat_idx;
	map_to_flat_idx.assign(measure_read.size(), 0);
	int cnt = 0;
	for (unsigned i = 0; i < measure_read.size(); i++)
		if (!b_measure_removed[i])
		{
			measure_read[cnt] = measure_read[i];
			map_to_flat_idx[i] = cnt++;
		}
	measure_read.resize(cnt);
	for (unsigned i = 0; i < measure_in_isoform.size(); i++)
	{
		cnt = 0;
		for (unsigned j = 0; j < measure_in_isoform[i].size(); j++)
		{
			int measure = measure_in_isoform[i][j];
			if (!b_measure_removed[measure])
			{
				measure_in_isoform[i][cnt] = map_to_flat_idx[measure];
				measure_virtual_length[i][cnt] = measure_virtual_length[i][j];
				cnt++;
			}
		}
		measure_in_isoform[i].resize(cnt);
		measure_virtual_length[i].resize(cnt);
	}
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  InferNew
// Description:  Given the all possible valid isoforms and other information, infer 
//               isoforms from scratch.
//  Parameters:  valid_isoforms  :  All possible valid isoforms, the infered isoforms 
//                   will only be generated from this set.
//--------------------------------------------------------------------------------------
void
IsoInferPE::InferNew (const vector<int>& set_sizes, 
                      const vector<double>& start_from_cnt, 
                      const vector<double>& end_at_cnt, 
					  const vector<vector<double> >& junc_cnt, 
					  const vector<vector<bool> >& valid_isoforms,
					  const vector<bool>& b_known_isoforms,
					  const vector<vector<int> >& start_exons, 
					  const vector<vector<int> >& end_exons, 
					  const vector<PEInfo>& pe_info,
					  vector<int>& solution)
{
	vector<int> exon_type;
	CalculateExonType(start_exons, end_exons, set_sizes.size(), exon_type);	
	if (set_sizes.size() < mPartitionSize)
		BestCombination(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, valid_isoforms, 
						b_known_isoforms, start_exons, end_exons, pe_info, true, solution);
	else
		PartitionAndSearch(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, valid_isoforms, 
							b_known_isoforms, start_exons, end_exons, pe_info, solution);
		
	return;
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  CalculateExonType
// Description:  Partition the whole instance into sub instances and combine the result.
//  Parameters:  
//               
//  	return:
//--------------------------------------------------------------------------------------
void
IsoInferPE::CalculateExonType( const vector<vector<int> >& start_exons, const vector<vector<int> >& end_exons, 
							   int exon_cnt, vector<int>& exon_type )
{
	// Calculate the type of each exon
	exon_type.assign(exon_cnt, 0);
	for (unsigned i = 0; i < start_exons.size(); i++)
		for (unsigned j = 0; j < start_exons[i].size(); j++)
			exon_type[start_exons[i][j]] = 1;

	for (unsigned i = 0; i < end_exons.size(); i++)
		for (unsigned j = 0; j < end_exons[i].size(); j++)
			if (1 == exon_type[end_exons[i][j]])
				exon_type[end_exons[i][j]] += 2;
}


//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  PartitionAndSearch
// Description:  Partition the whole instance into sub instances and combine the result.
//  Parameters:  
//               
//  	return:
//--------------------------------------------------------------------------------------
void
IsoInferPE::PartitionAndSearch (const vector<int>& set_sizes, 
								const vector<int>& exon_type,
								const vector<double>& start_from_cnt, 
								const vector<double>& end_at_cnt, 
								const vector<vector<double> >& junc_cnt, 
								const vector<vector<bool> >& valid_isoforms,	
								const vector<bool>& b_known_isoforms,
								const vector<vector<int> >& start_exons, 
								const vector<vector<int> >& end_exons, 
								const vector<PEInfo>& pe_info,
								vector<int>& solution)
{
	vector<vector<bool> > valid_isoforms_filtered = valid_isoforms;

	vector<vector<int> > sc_sets_init;
	int largest_ele = ConstructSetCover(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, 
								valid_isoforms, start_exons, end_exons, pe_info, true, sc_sets_init);

	vector<int> ele_covered_times;
	ele_covered_times.assign(largest_ele+1, 0);
	for (unsigned i = 0; i < sc_sets_init.size(); i++)
		for (unsigned j = 0; j < sc_sets_init[i].size(); j++)
			ele_covered_times[sc_sets_init[i][j]]++;

	vector<bool> removed_isoforms;
	removed_isoforms.assign(valid_isoforms.size(), false);

	vector<double> valid_isoforms_priority;
	valid_isoforms_priority.assign(valid_isoforms_filtered.size(), 0);

	int start_part_size = mPartitionSize-5;
	if (start_part_size < 5) start_part_size = 5;
	int end_part_size = mPartitionSize;
	if (end_part_size <= set_sizes.size()) end_part_size = set_sizes.size()-1;
	for (int part_size = start_part_size; part_size <= mPartitionSize; part_size++)
	{
		valid_isoforms_priority.assign(valid_isoforms_filtered.size(), 0);

		for (unsigned pro_start = 0; pro_start <= set_sizes.size() - part_size; pro_start++)
		{
			vector<int> set_sizes_pro;
			vector<double> start_from_cnt_pro;
			vector<double> end_at_cnt_pro;
			vector<int> exon_type_pro;
			vector<vector<double> > junc_cnt_pro;
			vector<vector<bool> > valid_isoforms_pro;
			vector<PEInfo> pe_info_pro;
			vector<vector<int> > valid_isoform_group;

			vector<int> sub_set;
			sub_set.resize(part_size);
			for (int i = 0; i < part_size; i++)
				sub_set[i] = pro_start + i;

			Project(set_sizes, set_sizes_pro,
					exon_type, exon_type_pro,
					start_from_cnt, start_from_cnt_pro,
					end_at_cnt, end_at_cnt_pro,
					junc_cnt, junc_cnt_pro,
					pe_info, pe_info_pro, sub_set);

			// For valid_isoforms_pro and valid_isoform_group
			Project(valid_isoforms_filtered, valid_isoforms_pro, sub_set, valid_isoform_group);
			vector<bool> b_known_isoforms_pro;
			b_known_isoforms_pro.assign(valid_isoform_group.size(), false);
			for (unsigned i = 0; i < valid_isoform_group.size(); i++)
				for (unsigned j = 0; j < valid_isoform_group[i].size(); j++)
					b_known_isoforms_pro[i] = b_known_isoforms_pro[i] | b_known_isoforms[valid_isoform_group[i][j]];
					
			vector<vector<int> > empty_vec;

			vector<int> curr_solution;
			BestCombination(set_sizes_pro, exon_type_pro, start_from_cnt_pro, end_at_cnt_pro, junc_cnt_pro, 
						valid_isoforms_pro, b_known_isoforms_pro, empty_vec, empty_vec, pe_info_pro, false, curr_solution);

			for (unsigned i = 0; i < curr_solution.size(); i++)
			{
				int curr_group = curr_solution[i];
				int curr_group_size = valid_isoform_group[curr_group].size();
				for (unsigned j = 0; j < valid_isoform_group[curr_group].size(); j++)
				{
					int curr_iso = valid_isoform_group[curr_group][j];
					valid_isoforms_priority[curr_iso] += 1.0 / curr_group_size;
					
					// Make sure that known isoforms will not be removed
					if (b_known_isoforms[curr_iso]) valid_isoforms_priority[curr_iso] += 2;
				}
			}
		}

		double min_prio = 1;

		// Remove low priority valid isoforms
		vector<int> remove_candidate;
		vector<double> remove_candidate_prio;
		for (unsigned i = 0; i < valid_isoforms_filtered.size(); i++)
			if (!removed_isoforms[i] && valid_isoforms_priority[i] < min_prio)
			{
				remove_candidate.push_back(i);
				remove_candidate_prio.push_back(valid_isoforms_priority[i]);
			}
		vector<int> sortedIdx;
		UtilityTemp<double, less<double> >::Sort(remove_candidate_prio, sortedIdx);
		UtilityTemp<int>::SortByIndex(remove_candidate, sortedIdx);

		int cnt = 0;
		for (unsigned i = 0; i < remove_candidate.size(); i++)
		{
			int curr_set = remove_candidate[i];
			// If current set does not make the original set cover infeasible, remove it.
			// otherwise, keep it.
			bool b_remove = true;
			for (unsigned j = 0; j < sc_sets_init[curr_set].size(); j++)
				if (ele_covered_times[sc_sets_init[curr_set][j]] <= 1)
				{
					b_remove = false;
					break;
				}

			if (b_remove)
			{
				cnt++;
				for (unsigned j = 0; j < sc_sets_init[curr_set].size(); j++)
					ele_covered_times[sc_sets_init[curr_set][j]]--;
				valid_isoforms_filtered[curr_set].assign(valid_isoforms_filtered[curr_set].size(), false);
				removed_isoforms[curr_set] = true;
			}
		}

		cout << cnt << " valid_isoforms are filtered out" << endl;
	}

	vector<vector<int> > sc_sets_final;
	vector<double> sc_set_weight_final;

	vector<int> idx_map;
	idx_map.resize(valid_isoforms_filtered.size());
	int cnt = 0;
	// Remove isoforms that have been filtered out.
	for (unsigned i = 0; i < valid_isoforms_filtered.size(); i++)
	{
		if (!removed_isoforms[i])
		{
			idx_map[cnt] = i;
			valid_isoforms_filtered[cnt] = valid_isoforms_filtered[i];
			mWeight[cnt] = mWeight[i];
			valid_isoforms_priority[cnt] = valid_isoforms_priority[i];
			cnt++;
		}
	}
	valid_isoforms_filtered.resize(cnt);

	largest_ele = ConstructSetCover(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, valid_isoforms_filtered, 
									start_exons, end_exons, pe_info, true, sc_sets_final);

	sc_set_weight_final.resize(mWeight.size());
	for (unsigned i = 0; i < mWeight.size(); i++)
		if (valid_isoforms_priority[i] != 0)
			sc_set_weight_final[i] = mWeight[i] / valid_isoforms_priority[i];
		else
			sc_set_weight_final[i] = 1000;  // In this case, valid_isoforms_filtered[i] will be empty.
	SolveSetCover(largest_ele, sc_set_weight_final, sc_sets_final, solution);

	for (unsigned i = 0; i < solution.size(); i++)
		solution[i] = idx_map[solution[i]];
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  Project
// Description:  Project the valid isoforms to a sub set of exons.
//  Parameters:  The exons are [pro_start, pro_end)
//               valid_isoform_group   :   The isoforms are grouped by the projection
//  	return:
//--------------------------------------------------------------------------------------
void
IsoInferPE::Project(const vector<vector<bool> >& isoforms, vector<vector<bool> >& isoforms_pro,
					const vector<int>& sub_set, vector<vector<int> >& isoform_group)
{
	isoforms_pro.clear();
	isoform_group.clear();
	int pro_size = sub_set.size();
	for (unsigned i = 0; i < isoforms.size(); i++)
	{
		vector<bool> iso_pro;
		iso_pro.resize(pro_size);
		for (int j = 0; j < sub_set.size(); j++)
			iso_pro[j] = isoforms[i][sub_set[j]];

		bool b_exist = false;
		unsigned idx = 0;
		for (idx = 0; idx < isoforms_pro.size(); idx++)
			if (iso_pro == isoforms_pro[idx])
			{
				b_exist = true;
				break;
			}

		if (b_exist)
			isoform_group[idx].push_back(i);
		else
		{
			vector<int> a_group;
			a_group.push_back(i);
			isoform_group.push_back(a_group);
			isoforms_pro.push_back(iso_pro);
		}
	}
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  Project
// Description:  Project the valid isoforms and related information to a sub set of exons.
//  Parameters:  The indexes of exons are sub_set
//               valid_isoform_group   :   The valid isoforms are grouped by the projection
//  	return:
//--------------------------------------------------------------------------------------
void
IsoInferPE::Project(const vector<int>& set_sizes, vector<int>& set_sizes_pro,
					const vector<int>& exon_type, vector<int>& exon_type_pro, 
		            const vector<double>& start_from_cnt, vector<double>& start_from_cnt_pro,
		            const vector<double>& end_at_cnt, vector<double>& end_at_cnt_pro,
					const vector<vector<double> >& junc_cnt, vector<vector<double> >& junc_cnt_pro,
					const vector<PEInfo>& pe_info, vector<PEInfo>& pe_info_pro,
					const vector<int>& sub_set)
{
	int pro_size = sub_set.size();
	set_sizes_pro.resize(pro_size);
	start_from_cnt_pro.resize(pro_size);
	end_at_cnt_pro.resize(pro_size);
	exon_type_pro.resize(pro_size);
	for (int i = 0; i < sub_set.size(); i++)
	{
		int idx = sub_set[i];
		set_sizes_pro[i] = set_sizes[idx];
		start_from_cnt_pro[i] = start_from_cnt[idx];
		end_at_cnt_pro[i] = end_at_cnt[idx];
		exon_type_pro[i] = exon_type[idx];
	}

	junc_cnt_pro.resize(pro_size);
	for (int i = 0; i < pro_size; i++)
	{
		junc_cnt_pro[i].resize(pro_size);
		for (int j = 0; j < pro_size; j++)
			junc_cnt_pro[i][j] = junc_cnt[sub_set[i]][sub_set[j]];
	}

	// For PE info
	pe_info_pro.resize(pe_info.size());
	for (unsigned i = 0; i < pe_info.size(); i++)
	{
		PEInfo& curr_info_pro= pe_info_pro[i];

		curr_info_pro.mSpanLow = pe_info[i].mSpanLow;
		curr_info_pro.mSpanHigh = pe_info[i].mSpanHigh;
		curr_info_pro.mSpanMean = pe_info[i].mSpanMean;
		curr_info_pro.mSpanStd = pe_info[i].mSpanStd;
		curr_info_pro.mReadLen = pe_info[i].mReadLen;
		curr_info_pro.mReadCnt = pe_info[i].mReadCnt;

		vector<vector<int> > part_comb_group;
		Project(pe_info[i].mPartComb, curr_info_pro.mPartComb, sub_set, part_comb_group);

		curr_info_pro.mPartCombDup.resize(part_comb_group.size());
		for (unsigned j = 0; j < part_comb_group.size(); j++)
		{
			int sum = 0;
			for (unsigned k = 0; k < part_comb_group[j].size(); k++)
				sum += pe_info[i].mPartCombDup[part_comb_group[j][k]];
			curr_info_pro.mPartCombDup[j] = sum;
		}
	}
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  SolveSetCover
// Description:  This method solve a set cover problem. 
//  Parameters:  sc_set_weight  : The weight of each set. 
//               sc_sets  :  Each element of the vector would be a set of the set cover
//                   instance. There are "valid_isoforms.size()" sets of the instance.
//                   The elements are [0, largest_ele]
//               solution :  The solution of the problem. This array contains all the 
//                   selected sets.
//      return:  The optimized objective value
//--------------------------------------------------------------------------------------
/*virtual*/
double
IsoInferPE::SolveSetCover (int largest_ele,
                           const vector<double>& sc_set_weight, 
						   const vector<vector<int> >& sc_sets,
						   vector<int>& solution)
{
//	// debug : output the problem
//	for (unsigned i = 0; i < sc_sets.size(); i++)
//	{
//		cout << i << "  :  " << sc_set_weight[i] << "  :  ";
//		for (unsigned j = 0; j < sc_sets[i].size(); j++)
//			cout << sc_sets[i][j] << ",";
//		cout << endl;
//	}
	vector<bool> is_measure_covered;
	is_measure_covered.assign(largest_ele+1, false);
	for (unsigned i = 0; i < sc_sets.size(); i++)
		for (unsigned j = 0; j < sc_sets[i].size(); j++)
			is_measure_covered[sc_sets[i][j]] = true;
	for (unsigned i = 0; i < is_measure_covered.size(); i++)
		if (!is_measure_covered[i])
		{
			solution.resize(sc_sets.size());
			for (unsigned i = 0; i < solution.size(); i++)
				solution[i] = i;
			cerr << "WARNING : This set cover problem is not feasible. largest_ele = " << largest_ele << endl;
			return 0;
		}

	glp_prob *blp;
	int row_cnt = largest_ele + 1;
	int col_cnt = sc_sets.size();
	int ele_cnt = row_cnt * col_cnt;
	int* im = new int[ele_cnt + 1];
	int* jm = new int[ele_cnt + 1];
	double* ar = new double[ele_cnt + 1];

	blp = glp_create_prob();
	//glp_term_out(false);
	glp_set_obj_dir(blp, GLP_MIN);
	glp_add_rows(blp, row_cnt);
	glp_add_cols(blp, col_cnt);

	// set row bound such that each element will be covered at least once.
	for (int i = 0; i < row_cnt; i++)
		glp_set_row_bnds(blp, i+1, GLP_LO, 1, 0.0);

	// set binary kind of all the variables
	for (int i = 0; i < col_cnt; i++)
		glp_set_col_kind(blp, i+1, GLP_BV);

	// set index and coefficients
	for (int i = 0; i < ele_cnt; i++)
	{
		im[i+1] = i / col_cnt + 1;
		jm[i+1] = i % col_cnt + 1;
		ar[i+1] = 0;
	}
	for (unsigned i = 0; i < sc_sets.size(); i++)
		for (unsigned j = 0; j < sc_sets[i].size(); j++)
		{
			int idx = sc_sets[i][j] * col_cnt + i + 1;
			im[idx] = sc_sets[i][j] + 1;
			jm[idx] = i + 1;
			ar[idx] = 1;
		}
	// set coefficients for the objective function
	for (int i = 0; i < col_cnt; i++)
		glp_set_obj_coef(blp, i + 1, sc_set_weight[i]);

	glp_load_matrix(blp, ele_cnt, im, jm, ar);

	glp_iocp parm;
	glp_init_iocp(&parm);
	parm.presolve = GLP_ON;
	glp_intopt(blp, &parm);

	double obj = glp_mip_obj_val(blp);

	// solution:
	for (int i = 0; i < col_cnt; i++)
		if (glp_mip_col_val(blp, i+1) > 0.5) solution.push_back(i);

	glp_delete_prob(blp);
	delete[] im;
	delete[] jm;
	delete[] ar;
	
	return obj;
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  StartEndInIsoform
// Description:  Given isoforms, and start-end pairs, find which start-end pair is covered
//               by each isoform.
//  Parameters:  start_end_pair_in_isoform.size() = 0 if the corresponding isoform is empty
//      return:  How many start-end exon pairs in all the isoforms.
//--------------------------------------------------------------------------------------
int
IsoInferPE::StartEndInIsoform (const vector<vector<bool> >& isoforms,
							   const vector<vector<int> >& start_exons, 
							   const vector<vector<int> >& end_exons, 
							   vector<vector<int> >& start_end_pair_in_isoform)
{
	if (isoforms.size() == 0) return 0;

	vector<int> start_exons_in_iso;
	vector<int> end_exons_in_iso;

	ExtractStartEndExons(isoforms, start_exons_in_iso, end_exons_in_iso);

	set<int> all_start_exons;
	set<int> all_end_exons;

	for (unsigned i = 0; i < start_exons.size(); i++)
	{
		for (unsigned j = 0; j < start_exons[i].size(); j++)
			all_start_exons.insert(start_exons[i][j]);
		for (unsigned j = 0; j < end_exons[i].size(); j++)
			all_end_exons.insert(end_exons[i][j]);
	}

	int exon_cnt = isoforms[0].size();

	start_end_pair_in_isoform.clear();
	start_end_pair_in_isoform.resize(isoforms.size());

	for (unsigned i = 0; i < isoforms.size(); i++)
	{
		vector<int>& av = start_end_pair_in_isoform[i];

		av.push_back(start_exons_in_iso[i]);
		av.push_back(end_exons_in_iso[i]);

		for (unsigned j = 0; j < start_exons.size(); j++)
		{
			bool b_succ = false;
			for (unsigned k = 0; k < start_exons[j].size(); k++)
				if (start_exons[j][k] == start_exons_in_iso[i])
				{
					b_succ = true;
					break;
				}
			if (!b_succ) continue;

			for (unsigned k = 0; k < end_exons[j].size(); k++)
				if (end_exons[j][k] == end_exons_in_iso[i])
				{
					b_succ = true;
					break;
				}
			if (!b_succ) continue;
			av.push_back(exon_cnt + j);
		}
	}

	vector<int> b_visited;
	b_visited.assign(exon_cnt + start_exons.size(), false);
	for (unsigned i = 0; i < start_end_pair_in_isoform.size(); i++)
		for (unsigned j = 0; j < start_end_pair_in_isoform[i].size(); j++)
			b_visited[start_end_pair_in_isoform[i][j]] = true;

	vector<int> map_to_flax_idx;
	map_to_flax_idx.resize(b_visited.size());
	int cnt = 0;
	for (unsigned i = 0; i < b_visited.size(); i++)
		if (b_visited[i])
			map_to_flax_idx[i] = cnt++;
	for (unsigned i = 0; i < start_end_pair_in_isoform.size(); i++)
		for (unsigned j = 0; j < start_end_pair_in_isoform[i].size(); j++)
			start_end_pair_in_isoform[i][j] = map_to_flax_idx[start_end_pair_in_isoform[i][j]];

	return cnt;
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ConstructSetCover
// Description:  Given the valid isoforms, the start, end exons and the part_comb, 
//               construct a set cover problem instance.
//               
//  Parameters:  b_enable_pe  :  Whether enable start-end exons or not.
//      return:  The largest element in the base set. The base set is [0, largest_ele].
//--------------------------------------------------------------------------------------
int
IsoInferPE::ConstructSetCover ( const vector<int>& set_sizes, 
								const vector<int>& exon_type,
								const vector<double>& start_from_cnt, 
								const vector<double>& end_at_cnt, 
								const vector<vector<double> >& junc_cnt, 
								const vector<vector<bool> >& isoforms,
							    const vector<vector<int> >& start_exons, 
							    const vector<vector<int> >& end_exons, 
								const vector<PEInfo>& pe_info,
								bool b_enable_se,
								vector<vector<int> >& sc_sets)
{
	vector<vector<int> >& measure_in_isoform = sc_sets;
	vector<vector<double> > measure_virtual_length;
	vector<double> measure_read;

	// Calculate the virtual lengths of exons and junctions.
	ExonJuncVirtualLength(set_sizes, exon_type, start_from_cnt, end_at_cnt, isoforms, junc_cnt, 
						measure_in_isoform, measure_read, measure_virtual_length);

	if (mbEnablePE)
	{
		// Calculate the virtual lengths of pairs or triples on all the valid isoforms
		vector<vector<int> > pair_triple_in_isoform;
		vector<vector<double> > pair_triple_virtual_length;
		vector<double> pair_triple_read;
		PairTripleVirtualLength(set_sizes, isoforms, pe_info, pair_triple_in_isoform, 
								pair_triple_read, pair_triple_virtual_length, true);

		CombineMeasure(measure_in_isoform, measure_virtual_length, measure_read,
				       pair_triple_in_isoform, pair_triple_virtual_length, pair_triple_read);

		// Remove measures that have no observed reads.
		vector<bool> b_measure_removed;
		b_measure_removed.assign(measure_read.size(), true);
		for (unsigned i = 0; i < measure_read.size(); i++)
			if (measure_read[i] > 0) b_measure_removed[i] = false;
		RemoveMeasures(measure_in_isoform, measure_virtual_length, measure_read, b_measure_removed);
	}

	
	if (b_enable_se)
	{
		vector<vector<int> > start_end_in_isoform;
		vector<vector<double> > start_end_virtual_length;
		vector<double> start_end_read;

		int se_cnt = StartEndInIsoform (isoforms, start_exons, end_exons, start_end_in_isoform);
		start_end_virtual_length.resize(isoforms.size());
		for (unsigned i = 0; i < start_end_virtual_length.size(); i++)
			start_end_virtual_length[i].assign(start_end_in_isoform[i].size(), 1);
		start_end_read.assign(se_cnt, 1);

		CombineMeasure(measure_in_isoform, measure_virtual_length, measure_read,
				       start_end_in_isoform, start_end_virtual_length, start_end_read);
	}

	return measure_read.size()-1;
}


/*virtual*/
void
IsoInferPE::CleanUp()
{
}

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  EnumerateValidByExpLevel
 *  Description:  Enumerate all the valid isoforms according to the junc reads and put the 
 *                result in valid_isoforms. The valid isoforms are enumerated in the order
 *                of the minimum number of junction read in isoforms.
 *        Param:  
 *       Return:  The isoforms are enumerated in the decreasing order of expression levels
 *--------------------------------------------------------------------------------------
 */
int
IsoInferPE::EnumerateValidByExpLevel(const vector<vector<double> >& junc_cnt, 
					       const vector<vector<int> >& start_exons, 
					       const vector<vector<int> >& end_exons, 
						   const vector<int>& set_sizes,
						   const vector<PEInfo>& pe_info,
						   vector<vector<bool> >& valid_isoforms,
						   vector<int>& valid_isoform_order)
{
	if (junc_cnt.size() == 0)
		return 0;

	GraphEx<int> valid_graph;
	int source;
	int sink;
	BuildSpliceGraph(valid_graph, junc_cnt, start_exons, end_exons, source, sink);

	// Enumerate in the order of the minimum number of junction reads in isoforms.
	vector<double> sorted_junc_cnt;
	vector<int> sorted_edge_id;
	for (unsigned i = 0; i < junc_cnt.size(); i++)
		for (unsigned j = i+1; j < junc_cnt.size(); j++)
		{
			int edge_id = valid_graph.EdgeID(valid_graph.GetNodeInID(i), valid_graph.GetNodeInID(j));
			if (-1 == edge_id) continue;
			sorted_junc_cnt.push_back(junc_cnt[i][j]);
			sorted_edge_id.push_back(edge_id);
		}

	vector<int> sortedIdx;
	UtilityTemp<double>::Sort(sorted_junc_cnt, sortedIdx);
	UtilityTemp<int>::SortByIndex(sorted_edge_id, sortedIdx);

	vector<int> grouped_edge_id;
	for (unsigned i = 0; i < sorted_edge_id.size(); i++)
		if (0 == i || sorted_junc_cnt[i] != sorted_junc_cnt[i-1])
			grouped_edge_id.push_back(i);
	grouped_edge_id.push_back(sorted_edge_id.size());

	int tot_node_cnt = valid_graph.NodeCnt(false);
	int tot_edge_cnt = valid_graph.EdgeCnt(false);
	bool* b_node_visited = new bool[tot_node_cnt];
	int* visited_nodes = new int[tot_node_cnt];
	int* visited_edges = new int[tot_edge_cnt];

	int order = 1;
	for (unsigned g = 0; g < grouped_edge_id.size()-1; g++)
	{
		// In case of isoforms containing at leas two exons
		for (int i = grouped_edge_id[g]; i < grouped_edge_id[g+1]; i++)
		{
			// Make sure that the isoforms enumerated in this round contains edge sorted_edge_id[i]
			int curr_edge_id = sorted_edge_id[i];

			// Mask all the edges with orders after the current edge
			vector<int> masked_edge;
			masked_edge.resize(sorted_edge_id.size()-i-1);
			for (int j = 0; j + i + 1 < sorted_edge_id.size(); j++)
				masked_edge[j] = sorted_edge_id[j+i+1];

			valid_graph.MaskAll(false);
			valid_graph.MaskEdge(masked_edge, true);

			// Find out all the edges that can be reached from the end point
			// of the current edge and Find out all the edges that can be reached reversely
			// from the start point of the current edge.
			for (int i = 0; i < tot_node_cnt; i++)
				b_node_visited[i] = false;
			int node_visited_cnt = 0;
			int edge_visited_cnt = 0;
			GraphAlgorithm_Basic::DFS(&valid_graph, valid_graph.ToNode(curr_edge_id), b_node_visited, 
									  visited_nodes, &node_visited_cnt, NULL, visited_edges, &edge_visited_cnt);

//			cout << "effective edges = " << edge_visited_cnt << "  " << endl;
//			for (int i = 0; i < edge_visited_cnt; i++)
//				cout << visited_edges[i] << "\t";
//			cout << endl;

			GraphAlgorithm_Basic::DFSReverse(&valid_graph, valid_graph.FromNode(curr_edge_id), b_node_visited, 
									  visited_nodes, &node_visited_cnt, NULL, visited_edges, &edge_visited_cnt);
			visited_edges[edge_visited_cnt++] = curr_edge_id;

//			cout << "effective edges = " << edge_visited_cnt << "  " << endl;
//			for (int i = 0; i < edge_visited_cnt; i++)
//				cout << visited_edges[i] << "\t";
//			cout << endl;

			// Keep all the visited edges and current edge
			valid_graph.MaskAll(true);
			valid_graph.MaskNode(visited_nodes, node_visited_cnt, false, false);
			valid_graph.MaskEdge(visited_edges, edge_visited_cnt, false);


			EnumerateValid(valid_graph, source, sink, start_exons, end_exons, set_sizes, pe_info, valid_isoforms);
		}

		unsigned old_size = valid_isoform_order.size();
		order++;
		valid_isoform_order.resize(valid_isoforms.size());
		for (unsigned j = old_size; j < valid_isoforms.size(); j++)
			valid_isoform_order[j] = order;
		cout << "valid_isoforms.size() = " << valid_isoforms.size() 
					 << " min junction read = " << sorted_junc_cnt[grouped_edge_id[g]] << endl;
		if (valid_isoforms.size() > mMaxValidIsoformCnt) 
		{
			valid_isoforms.resize(mMaxValidIsoformCnt);
			mOmittedJunctionCnt = sorted_junc_cnt[grouped_edge_id[g]];
			cout << "Part of junctions with reads less than " << mOmittedJunctionCnt << " are omitted" << endl;
			break;
		}
	}

	delete[] b_node_visited;
	delete[] visited_edges;

	unsigned old_size = valid_isoform_order.size();
	// In case of isoforms containing a single exon
	valid_graph.MaskAllEdges(false);
	valid_graph.MaskEdge(sorted_edge_id, true);
	EnumerateValid(valid_graph, source, sink, start_exons, end_exons, set_sizes, pe_info, valid_isoforms);

	valid_isoform_order.resize(valid_isoforms.size());
	for (unsigned i = old_size; i < valid_isoforms.size(); i++)
		valid_isoform_order[i] = order;

	return valid_isoforms.size();
}		/* -----  end of method IsoInferPE::EnumerateValidByExpLevel  ----- */

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  EnumerateValid
 *  Description:  Enumerate all the valid seeds according to the junc reads
 *        Param:  node   :   The internal ID of current node
 *                sink   :   The internal ID of the sink
 *       Return:  the last elements of existing_isoform will be the seed
 *--------------------------------------------------------------------------------------
 */
void
IsoInferPE::EnumerateValid( GraphEx<int>& valid_graph, 
							int source, int sink, 
							const vector<vector<int> >& start_exons, 
							const vector<vector<int> >& end_exons, 
							const vector<int>& set_sizes,
							const vector<PEInfo>& pe_info,
							vector<vector<bool> > & valid_isoforms)
{
	// Maybe some edges related to the start or end exons are masked
	vector<vector<int> > effective_start_exons_in_id;
	vector<vector<int> > effective_end_exons_in_id;
	for (unsigned i = 0; i < start_exons.size(); i++)
	{
		vector<int> eff_start;
		for (unsigned j = 0; j < start_exons[i].size(); j++)
		{
			int in_id = valid_graph.GetNodeInID(start_exons[i][j]);
			if (valid_graph.InDegree(in_id) > 0)
				eff_start.push_back(in_id);
		}
		vector<int> eff_end;
		for (unsigned j = 0; j < end_exons[i].size(); j++)
		{
			int in_id = valid_graph.GetNodeInID(end_exons[i][j]);
			if (valid_graph.OutDegree(in_id) > 0)
				eff_end.push_back(in_id);
		}
		if (eff_start.size() != 0 && eff_end.size() != 0)
		{
			effective_start_exons_in_id.push_back(eff_start);
			effective_end_exons_in_id.push_back(eff_end);
		}
	}

	int tot_node_cnt = valid_graph.NodeCnt(false);
	bool* b_node_visited = new bool[tot_node_cnt];
	int* visited_nodes = new int[tot_node_cnt];

	vector<int> init_unmasked_edges;
	vector<int> init_unmasked_nodes;
	const int* edges = valid_graph.Edges();
	for (int i = 0; i < valid_graph.EdgeCnt(); i++)
		init_unmasked_edges.push_back(edges[i]);
	const int* nodes = valid_graph.Nodes();
	for (int i = 0; i < valid_graph.NodeCnt(); i++)
		init_unmasked_nodes.push_back(nodes[i]);

	// For each start-end exon pair, mask all the edges that would not lead to
	for (unsigned i = 0; i < effective_start_exons_in_id.size(); i++)
	{
		//cout << effective_start_exons_in_id[i][0] << " , " << effective_end_exons_in_id[i][0] << endl;
		set<int> starts;
		for (unsigned j = 0; j < effective_start_exons_in_id[i].size(); j++)
			starts.insert(effective_start_exons_in_id[i][j]);
		set<int> ends;
		for (unsigned j = 0; j < effective_end_exons_in_id[i].size(); j++)
			ends.insert(effective_end_exons_in_id[i][j]);

		set<int> masked_edges;
		// Disconnect all other starts from the source and ends from the sink
		const int* out_edges = valid_graph.OutEdges(source);
		for (int j = 0; j < valid_graph.OutDegree(source); j++)
			if (starts.find(valid_graph.ToNode(out_edges[j])) == starts.end())
				masked_edges.insert(out_edges[j]);

		const int* in_edges = valid_graph.InEdges(sink);
		for (int j = 0; j < valid_graph.InDegree(sink); j++)
			if (ends.find(valid_graph.FromNode(in_edges[j])) == ends.end())
				masked_edges.insert(in_edges[j]);

		vector<int> kept_edges;
		for (unsigned j = 0; j < init_unmasked_edges.size(); j++)
			if (masked_edges.find(init_unmasked_edges[j]) == masked_edges.end())
				kept_edges.push_back(init_unmasked_edges[j]);

//		cout << "Kept edges  = " << kept_edges.size()  << "  " << endl;
//		for (int j = 0; j < kept_edges.size(); j++)
//			cout << kept_edges[j] << ",";
//		cout << endl;

		vector<int> removed_edges;
		removed_edges.assign(masked_edges.begin(), masked_edges.end());
		valid_graph.MaskEdge(removed_edges, true);

		//cout << "Unmasked node 1 = " << valid_graph.NodeCnt() << endl;
		//cout << "Unmasked edge 1 = " << valid_graph.EdgeCnt() << endl;

		// Find out reachable nodes from the source
		for (int j = 0; j < tot_node_cnt; j++)
			b_node_visited[j] = false;

		int node_visited_cnt = 0;
		int edge_visited_cnt = 0;
		GraphAlgorithm_Basic::DFS(&valid_graph, source, b_node_visited, visited_nodes, 
								  &node_visited_cnt, NULL, NULL, &edge_visited_cnt);

		//cout << "Reachable from source : ";
		set<int> reachable_nodes_from_source;
		for (int j = 0; j < node_visited_cnt; j++)
		{
			//cout << visited_nodes[j] << ",";
			reachable_nodes_from_source.insert(visited_nodes[j]);
		}
		//cout << endl;

		// Find out nodes which can get to the sink
		for (int j = 0; j < tot_node_cnt; j++)
			b_node_visited[j] = false;
		node_visited_cnt = 0;
		edge_visited_cnt = 0;
		GraphAlgorithm_Basic::DFSReverse(&valid_graph, sink, b_node_visited, visited_nodes, 
										 &node_visited_cnt, NULL, NULL, &edge_visited_cnt);

		//cout << "Reachable to sink : ";
		vector<int> good_nodes;
		for (int j = 0; j < node_visited_cnt; j++)
		{
			//cout << visited_nodes[j] << ",";
			if (reachable_nodes_from_source.find(visited_nodes[j]) != reachable_nodes_from_source.end())
				good_nodes.push_back(visited_nodes[j]);
		}
		//cout << endl;

		//cout << " Total " << good_nodes.size() << " good nodes" << endl;

		valid_graph.MaskAll(true);
		valid_graph.MaskNode(good_nodes, false, false);
		valid_graph.MaskEdge(kept_edges, false);

		vector<bool> b_visited;
		b_visited.assign(set_sizes.size() + 2, false);

		vector<bool> an_iso;
		an_iso.assign(set_sizes.size(), false);

		EnumerateValidDFS(valid_graph, source, sink, set_sizes, pe_info, b_visited, an_iso, valid_isoforms);

		if (valid_isoforms.size() > mMaxValidIsoformCnt)
			break;

		valid_graph.MaskAll(true);
		valid_graph.MaskNode(init_unmasked_nodes, false);
		valid_graph.MaskEdge(init_unmasked_edges, false);
	}

	delete[] b_node_visited;
	delete[] visited_nodes;
}

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  EnumerateValid
 *  Description:  Enumerate all the valid seeds according to the junc reads
 *        Param:  node   :   The internal ID of current node
 *                sink   :   The internal ID of the sink
 *       Return:  
 *--------------------------------------------------------------------------------------
 */
void
IsoInferPE::EnumerateValidDFS(GraphEx<int>& valid_graph, 
							int node, int sink, 
							const vector<int>& set_sizes,
							const vector<PEInfo>& pe_info,
							vector<bool>& b_visited, 
							vector<bool>& an_iso,
							vector<vector<bool> > & valid_isoforms)
{
	if (node == sink)
	{
		valid_isoforms.push_back(an_iso);
		return;
	}

	b_visited[node] = true;
	int curr_exon = valid_graph.GetNodeExID(node);
	if (curr_exon >= 0 && curr_exon < an_iso.size())  // not source
		an_iso[curr_exon] = true;

	bool b_pass = true;

	// Check the partial iso
	if (curr_exon >= 0 && mbEnablePE)
	{
		int max_gap = 0;
		for (unsigned i = 0; i < pe_info.size(); i++)
			if (max_gap < pe_info[i].mSpanHigh)
				max_gap = pe_info[i].mSpanHigh;

		// pair check
		int gap = 0;
		for (int start = curr_exon-1; start >= 0; start--)
		{
			if (!an_iso[start]) continue;

			// Check whether this part comb is supported by the pe_info
			if (!IsPairSupported(set_sizes, start, curr_exon, gap, pe_info))
				b_pass = false;
			
			gap += set_sizes[start];
			if (gap >= max_gap) break;
		}

		// triple check
		// === === ................ ====
		// pair_start last_one ....... curr_exon
		gap = 0;
		int pair_start = curr_exon;
		while (b_pass)
		{
			int last_one = pair_start;
			pair_start--;
			while (pair_start >= 0 && !an_iso[pair_start])
				pair_start--;
			if (pair_start < 0) break;
			if (curr_exon == last_one) continue;
			gap += set_sizes[last_one];
			if (gap >= max_gap) break;

			if (!IsTripleSupported(set_sizes, pair_start, last_one, curr_exon, gap, pe_info))
				b_pass = false;
		}
		// === .................. === ====
		// pair_start ..... precessor curr_exon
		gap = 0;
		pair_start = curr_exon;
		int precessor = curr_exon;
		while (b_pass)
		{
			pair_start--;
			while (pair_start >= 0 && !an_iso[pair_start])
				pair_start--;
			if (pair_start < 0) break;
			if (precessor == curr_exon)
			{ 
				precessor = pair_start;
				gap += set_sizes[precessor];
				continue;
			}

			if (!IsTripleSupported(set_sizes, curr_exon, precessor, pair_start, gap, pe_info))
				b_pass = false;

			gap += set_sizes[pair_start];
			if (gap >= max_gap) break;
		}
	} // if (mbEnablePE)

	if (b_pass)
	{
		const int* out_edges = valid_graph.OutEdges(node);
		for (int i = 0; i < valid_graph.OutDegree(node); i++)
		{
			int to_node = valid_graph.ToNode(out_edges[i]);
			EnumerateValidDFS(valid_graph, to_node, sink, set_sizes, pe_info, b_visited, an_iso, valid_isoforms);
			if (valid_isoforms.size() > mMaxValidIsoformCnt) 
				break;
		}
	}

	b_visited[node] = false;
	if (curr_exon >= 0 && curr_exon < an_iso.size())  // not source
		an_iso[curr_exon] = false;
}		/* -----  end of method IsoInferPE::EnumerateValidDFS  ----- */

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  CheckStartEnd
 *  Description:  Check the start and end pair.
 *        Param:  a_valid_isoform : A valid isoform. This function assume that the isoform
 *                    is valid.
 *                start_exons : The exons that serve as the start exons
 *                end_exons : The exons that serve as the end exons. Note that the 
 *                    start exons and end_exons are paired.
 *                pe_len    : The length of PE reads
 *       Return:  whether this isoform is supported ty the PE information or not.
 *--------------------------------------------------------------------------------------
 */
bool
IsoInferPE::CheckStartEnd(const vector<bool>& a_valid_isoform, 
		                  const vector<vector<int> >& start_exons, 
						  const vector<vector<int> >& end_exons)
{
	// Check start and end exons
	assert(start_exons.size() == end_exons.size());
	int set_cnt = a_valid_isoform.size();
	int s_exon = -1;
	int e_exon = -1;
	for (int i = 0; i < set_cnt; i++)
	{
		if (a_valid_isoform[i])
		{
			s_exon = i;
			break;
		}
	}
	for (int i = set_cnt - 1; i >= 0; i--)
	{
		if (a_valid_isoform[i])
		{
			e_exon = i;
			break;
		}
	}

	if (-1 == s_exon || -1 == e_exon) return false;

	bool b_succ = false;
	for (unsigned i = 0; i < start_exons.size(); i++)
	{
		b_succ = false;
		for (unsigned j = 0; j < start_exons[i].size(); j++)
			if (s_exon == start_exons[i][j])
			{
				b_succ = true;
				break;
			}
		if (!b_succ) continue;
		b_succ = false;
		for (unsigned j = 0; j < end_exons[i].size(); j++)
			if (e_exon == end_exons[i][j])
			{
				b_succ = true;
				break;
			}
		if (b_succ) break;
	}

	return b_succ;
}		/* -----  end of method IsoInferPE::CheckStartEnd  ----- */

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  MaxConsistExp
 *  Description:  Given an isoform and the paired-end information, calculate the maximum
 *                expression level, above which this isoform will not pass the consistency
 *                checking.
 *        Param:  
 *       Return:  The expression level cutoff
 *--------------------------------------------------------------------------------------
 */
double
IsoInferPE::MaxConsistExp(const vector<bool>& a_valid_isoform, const vector<int>& set_sizes, const vector<PEInfo>& pe_info)
{
	int set_cnt = a_valid_isoform.size();

	int max_gap = 0;
	for (unsigned i = 0; i < pe_info.size(); i++)
		if (max_gap < pe_info[i].mSpanHigh)
			max_gap = pe_info[i].mSpanHigh;

	// Find the max expression level that is lead by unsupported pairs / triples
	double max_exp_in_rpkm = 100000;

	// Extract all the existing paired partial combination of current isoform.
	for (int start = 0; start < set_cnt; start++)
	{
		if (!a_valid_isoform[start]) continue;
		for (unsigned end = start+1; end < set_sizes.size(); end++)
		{
			if (!a_valid_isoform[end]) continue;
			int gap = 0;
			for (unsigned idx = start + 1; idx <= end - 1; idx++)
				if (a_valid_isoform[idx]) gap += set_sizes[idx];

			vector<bool> part_comb;
			part_comb.assign(set_sizes.size(), false);
			part_comb[start] = true;
			part_comb[end] = true;

			if (!IsPartCombSupported(part_comb, pe_info))
			{
				double vir_len_sum = 0;
				for (unsigned i = 0; i < pe_info.size(); i++)
				{
					double vir_len = VirtualLength(pe_info[i], set_sizes[start], gap, set_sizes[end]);
					vir_len_sum += vir_len * pe_info[i].mReadCnt / 1000000000.0;
				}

				double exp_in_rpkm = log(1/mConfidenceLevel) / vir_len_sum;
				if (max_exp_in_rpkm > exp_in_rpkm) max_exp_in_rpkm = exp_in_rpkm;
			}
			
			if (gap >= max_gap) break;
		}
	}

	// Extract all the existing triple partial combination of current isoform.
	int pair_start = -1;
	while (true)
	{
		int last_one = pair_start;
		pair_start++;
		while (pair_start < set_cnt && !a_valid_isoform[pair_start])
			pair_start++;
		if (pair_start == set_cnt) break;

		if (-1 == last_one) continue;

		for (unsigned end = pair_start + 1; end < set_sizes.size(); end++)
		{
			// Calculate the max expressin level such that this triple is not informative
			if (!a_valid_isoform[end]) continue;
			int gap = set_sizes[pair_start];
			for (unsigned idx = pair_start + 1; idx <= end - 1; idx++)
				if (a_valid_isoform[idx]) gap += set_sizes[idx];

			vector<bool> part_comb;
			part_comb.assign(set_sizes.size(), false);
			part_comb[last_one] = true;
			part_comb[pair_start] = true;
			part_comb[end] = true;

			if (!IsPartCombSupported(part_comb, pe_info))
			{
				double vir_len_sum = 0;
				for (unsigned i = 0; i < pe_info.size(); i++)
				{
					double vir_len = VirtualLength(pe_info[i], pe_info[i].mReadLen, gap, set_sizes[end]);
					vir_len_sum += vir_len * pe_info[i].mReadCnt / 1000000000.0;
				}

				double exp_in_rpkm = log(1/mConfidenceLevel) / vir_len_sum;
				if (max_exp_in_rpkm > exp_in_rpkm) max_exp_in_rpkm = exp_in_rpkm;
			}

			if (gap >= max_gap) break;
		}

		for (int end = last_one - 1; end >= 0; end--)
		{
			if (!a_valid_isoform[end]) continue;
			int gap = set_sizes[last_one];
			for (int idx = end + 1; idx <= last_one - 1; idx++)
				if (a_valid_isoform[idx]) gap += set_sizes[idx];

			vector<bool> part_comb;
			part_comb.assign(set_sizes.size(), false);
			part_comb[last_one] = true;
			part_comb[pair_start] = true;
			part_comb[end] = true;

			if (!IsPartCombSupported(part_comb, pe_info))
			{
				double vir_len_sum = 0;
				for (unsigned i = 0; i < pe_info.size(); i++)
				{
					double vir_len = VirtualLength(pe_info[i], pe_info[i].mReadLen, gap, set_sizes[end]);
					vir_len_sum += vir_len * pe_info[i].mReadCnt / 1000000000.0;
				}

				double exp_in_rpkm = log(1/mConfidenceLevel) / vir_len_sum;
				if (max_exp_in_rpkm > exp_in_rpkm) max_exp_in_rpkm = exp_in_rpkm;
			}

			if (gap >= max_gap) break;
		}
	}

	return max_exp_in_rpkm;
}

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  CheckPEConsistency
 *  Description:  Given an isoform and the paired-end information, check whether this 
 *                isoform is supported by the PE information
 *        Param:  a_valid_isoform : A valid isoform. This function assume that the isoform
 *                    is valid.
 *                start_exons : The exons that serve as the start exons
 *                end_exons : The exons that serve as the end exons. Note that the 
 *                    start exons and end_exons are paired.
 *                pe_len    : The length of PE reads
 *       Return:  whether this isoform is supported ty the PE information or not.
 *--------------------------------------------------------------------------------------
 */
bool
IsoInferPE::CheckPEConsistency(const vector<bool>& a_valid_isoform, const vector<int>& set_sizes, const vector<PEInfo>& pe_info)
{
	int set_cnt = a_valid_isoform.size();

	int max_gap = 0;
	for (unsigned i = 0; i < pe_info.size(); i++)
		if (max_gap < pe_info[i].mSpanHigh)
			max_gap = pe_info[i].mSpanHigh;

	// Extract all the existing paired partial combination of current isoform.
	for (int start = 0; start < set_cnt; start++)
	{
		if (!a_valid_isoform[start]) continue;
		for (unsigned end = start+1; end < set_sizes.size(); end++)
		{
			if (!a_valid_isoform[end]) continue;
			int gap = 0;
			for (unsigned idx = start + 1; idx <= end - 1; idx++)
				if (a_valid_isoform[idx]) gap += set_sizes[idx];

			// Check whether this part comb is supported by the pe_info
			if (!IsPairSupported(set_sizes, start, end, gap, pe_info))
				return false;
			
			if (gap >= max_gap) break;
		}
	}

	// Extract all the existing triple partial combination of current isoform.
	int pair_start = -1;
	while (true)
	{
		int last_one = pair_start;
		pair_start++;
		while (pair_start < set_cnt && !a_valid_isoform[pair_start])
			pair_start++;
		if (pair_start == set_cnt) break;

		if (-1 == last_one) continue;

		for (unsigned end = pair_start + 1; end < set_sizes.size(); end++)
		{
			if (!a_valid_isoform[end]) continue;
			int gap = set_sizes[pair_start];
			for (unsigned idx = pair_start + 1; idx <= end - 1; idx++)
				if (a_valid_isoform[idx]) gap += set_sizes[idx];

			// Check whether this part comb is supported by the pe_info
			if (!IsTripleSupported(set_sizes, last_one, pair_start, end, gap, pe_info))
				return false;

			if (gap >= max_gap) break;
		}

		for (int end = last_one - 1; end >= 0; end--)
		{
			if (!a_valid_isoform[end]) continue;
			int gap = set_sizes[last_one];
			for (int idx = end + 1; idx <= last_one - 1; idx++)
				if (a_valid_isoform[idx]) gap += set_sizes[idx];

			// Check whether this part comb is supported by the pe_info
			if (!IsTripleSupported(set_sizes, last_one, pair_start, end, gap, pe_info))
				return false;

			if (gap >= max_gap) break;
		}
	}

	return true;
}

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  IsInformative
 *  Description:  Whether a given pair or triple is informative
 *
 *                ==============            ===============
 *                <-size of e1-><----gap---><-size of e2-->
 *        Param:  len_exon1   :  If a triple is tested, pass 0 to this parameter.
 *       Return:  
 *         Note:  
 *--------------------------------------------------------------------------------------
 */
bool
IsoInferPE::IsInformative(int len_exon1, int len_exon2, int gap, const vector<PEInfo>& pe_info)
{
	double exp_in_read_cnt = 0;
	for (unsigned i = 0; i < pe_info.size(); i++)
	{
		const PEInfo& curr_pe_info = pe_info[i];
		int read_len = pe_info[i].mReadLen;
		double vir_len = VirtualLength(curr_pe_info, len_exon1 + read_len, gap, len_exon2 + read_len);
		exp_in_read_cnt += vir_len / 1000 * mMinExpLevel * curr_pe_info.mReadCnt / 1000000;
	}

	// Calculate the probability of not observing any reads
	double prob = 1 / exp(exp_in_read_cnt);

	if (prob < mConfidenceLevel)
		return true;

	return false;
}


/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  IsPartCombSupported
 *  Description:  Given a partial combination and the pe info, decide whether this 
 *                partial combination is supported or not.
 *
 *        Param:  
 *       Return:  
 *         Note: 
 *--------------------------------------------------------------------------------------
 */
bool
IsoInferPE::IsPartCombSupported(const vector<bool>& part_comb, const vector<PEInfo>& pe_info)
{
	for (unsigned i = 0; i < pe_info.size(); i++)
	{
		const PEInfo& curr_pe_info = pe_info[i];

		for (unsigned j = 0; j < curr_pe_info.mPartComb.size(); j++)
		{
			bool b_succ = true;
			for (unsigned k = 0; k < part_comb.size(); k++)
				if (part_comb[k] && !curr_pe_info.mPartComb[j][k])
				{
					b_succ = false;
					break;
				}
			if (b_succ)
			{
				if (curr_pe_info.mPartCombDup[j] >= 1)
					return true;
			}
		}

	}

	return false;
}

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  IsPairSupported
 *  Description:  Given the length of two exons, the gap between the two exons, the 
 *                allowed gap lower and upper bound, decide whether this pair combination
 *                should be considered, calculate the probability of 
 *                observing at least a certain number of PE reads and decide whether
 *                this pair combination is supported by the info
 *
 *                ==============            ===============
 *                <-size of e1-><----gap---><-size of e2-->
 *        Param:  exon1  :  The first exon
 *       Return:  
 *         Note:  This function also needs the following three member variables:
 *                mMinReadOnPartComb   :   The minimum number of allowed reads on a part comb
 *                mMinExpLevel         :   The minimum expression level detectable
 *--------------------------------------------------------------------------------------
 */
bool
IsoInferPE::IsPairSupported(const vector<int>& set_sizes, int exon1, int exon2, int gap, const vector<PEInfo>& pe_info)
{
	vector<bool> part_comb;
	part_comb.assign(set_sizes.size(), false);
	part_comb[exon1] = true;
	part_comb[exon2] = true;

	if (IsPartCombSupported(part_comb, pe_info))
		return true;
	if (IsInformative(set_sizes[exon1], set_sizes[exon2], gap, pe_info))
		return false;
	return true;
}

/*
 *--------------------------------------------------------------------------------------
 *        Class:  IsoInferPE
 *       Method:  IsTripleSupported
 *  Description:  Decide whether a triple combination
 *                should be considered, calculate the probability of 
 *                observing at least a certain number of PE reads and decide whether
 *                this pair combination is supported by the info
 *
 *                ==============            ===============
 *                <-size of e1-><----gap---><-size of e2-->
 *        Param:  exon1  :  The first exon that forms the junction
 *                exon2  :  The second exon that forms the junction
 *                exon   :  The third exon.
 *
 *       Return:  
 *         Note:  This function also needs the following three member variables:
 *                mMinReadOnPartComb   :   The minimum number of allowed reads on a part comb
 *                mMinExpLevel         :   The minimum expression level detectable
 *--------------------------------------------------------------------------------------
 */
bool
IsoInferPE::IsTripleSupported(const vector<int>& set_sizes, int exon1, int exon2, int exon, int gap, const vector<PEInfo>& pe_info)
{
	vector<bool> part_comb;
	part_comb.assign(set_sizes.size(), false);
	part_comb[exon1] = true;
	part_comb[exon2] = true;
	part_comb[exon] = true;

	if (IsPartCombSupported(part_comb, pe_info))
		return true;
	if (IsInformative(0, set_sizes[exon], gap, pe_info))
		return false;
	return true;
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  ExonJuncVirtualLength
// Description:  For exons and junctions on each isoform, calculate the virtual length.
//  Parameters: 
//--------------------------------------------------------------------------------------
void
IsoInferPE::ExonJuncVirtualLength(const vector<int>& set_sizes, 
								  const vector<int>& exon_type,
								  const vector<double>& start_from_cnt, 
								  const vector<double>& end_at_cnt, 
								  const vector<vector<bool> >& isoforms, 
								  const vector<vector<double> >& junc_cnt, 
								  vector<vector<int> >& exon_junc_in_isoform, 
								  vector<double>& exon_junc_read, 
								  vector<vector<double> >& exon_junc_virtual_length)
{
	if (isoforms.size() == 0)
	{
		exon_junc_in_isoform.clear();
		exon_junc_read.clear();
		exon_junc_virtual_length.clear();
		return;
	}
	int read_len = mpInstance->mReadLen;
	int cross_strength = mpInstance->mCrossStrength;
	int junc_len = read_len - cross_strength + 1;
	unsigned set_cnt = set_sizes.size();

	vector<double> adjusted_set_sizes;
	adjusted_set_sizes.resize(set_sizes.size());
	vector<double> adjusted_sample_cnt;
	adjusted_sample_cnt.resize(set_sizes.size());
	for (unsigned i = 0; i < set_sizes.size(); i++)
	{
		adjusted_set_sizes[i] = set_sizes[i];
		if (0 == exon_type[i])         // Not start/end exon
			adjusted_sample_cnt[i] = (start_from_cnt[i] + end_at_cnt[i]) / 2.0;
		else if (1 == exon_type[i])    // Start exon
			adjusted_sample_cnt[i] = start_from_cnt[i];
		else if (2 == exon_type[i])    // End exon
			adjusted_sample_cnt[i] = end_at_cnt[i];
		else if (3 == exon_type[i])    // Start and end exon
		{
			adjusted_set_sizes[i] = set_sizes[i] - (read_len + cross_strength) / 2.0 + 1;
			adjusted_sample_cnt[i] = (start_from_cnt[i] + end_at_cnt[i]) / 2.0;
		}
	}

	// In the following, the virtual lengths of some junctions are
	// set to -1. Such junctions should not be used in the LP 
	// calculation. However, they are necessary for the set cover
	// problem.
	exon_junc_in_isoform.resize(isoforms.size());
	exon_junc_virtual_length.resize(isoforms.size());
	exon_junc_read.assign(set_cnt * set_cnt + set_cnt, -1);
	for (unsigned i = 0; i < isoforms.size(); i++)
	{
		exon_junc_in_isoform[i].clear();
		exon_junc_virtual_length[i].clear();
		for (int j = 0; j < set_cnt; j++)
			if (isoforms[i][j])
			{
				double v_len = adjusted_set_sizes[j];
				exon_junc_in_isoform[i].push_back(j);
				exon_junc_virtual_length[i].push_back(v_len);
				exon_junc_read[j] = adjusted_sample_cnt[j];
			}

		vector<int> prev_len;
		vector<int> succ_len;
		prev_len.assign(set_sizes.size(), 0);
		succ_len.assign(set_sizes.size(), 0);

		for (int j = 1; j < set_cnt; j++)
			if (isoforms[i][j-1]) prev_len[j] = set_sizes[j-1] + prev_len[j-1];
			else prev_len[j] = prev_len[j-1];
		for (int j = set_cnt-2; j >= 0; j--)
			if (isoforms[i][j+1]) prev_len[j] += set_sizes[j+1] + prev_len[j+1];
			else prev_len[j] = prev_len[j+1];

		int start = -1;
		for (int j = 0; j < set_cnt; j++)
		{
			if (isoforms[i][j])
			{
				if (-1 != start)
				{
					int min_len = junc_len;
					if (min_len > prev_len[start])
						min_len = prev_len[start];
					if (min_len > succ_len[start] + set_sizes[start])
						min_len = succ_len[start] + set_sizes[start];

					int idx = start * set_cnt + j + set_cnt;
					exon_junc_in_isoform[i].push_back(idx);
					exon_junc_virtual_length[i].push_back(min_len);
					exon_junc_read[idx] = junc_cnt[start][j];
				}
				start = j;
			}
		}
	}

	// Remove empty measure 
	vector<bool> b_measure_removed;
	b_measure_removed.assign(exon_junc_read.size(), false);
	for (unsigned i = 0; i < exon_junc_read.size(); i++)
		if (-1 == exon_junc_read[i]) b_measure_removed[i] = true;

	RemoveMeasures(exon_junc_in_isoform, exon_junc_virtual_length, exon_junc_read, b_measure_removed);
}


//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  PairTripleVirtualLength
// Description:  For triples or pairs on each isoform, calculate the virtual length.
//  Parameters:  For different PE info, the virtual length is a weighted sum.
//               pair_triple_read is the observed number of reads for each pair/triple
//               b_allpt  :  When a pair / triple contains an exon that is the last one
//                   in an isoform, the number of expected reads on such pair/triple 
//                   does not follow the same distribution as other pair/triples. If
//                   b_allpt is true, such pair/triples will be included. Otherwise,
//                   no such pair/triples will be considered.
//--------------------------------------------------------------------------------------
void
IsoInferPE::PairTripleVirtualLength(const vector<int>& set_sizes, const vector<vector<bool> >& isoforms, 
									const vector<PEInfo>& pe_info, vector<vector<int> >& pair_triple_in_isoform, 
									vector<double>& pair_triple_read, vector<vector<double> >& pair_triple_virtual_length,
									bool b_allpt)
{
	vector<map_64_double> virtual_length;

	virtual_length.resize(isoforms.size());
	for (unsigned i = 0; i < isoforms.size(); i++)
		for (unsigned j = 0; j < pe_info.size(); j++)
			VirtualLength(isoforms[i], set_sizes, pe_info[j], virtual_length[i]);

	int set_cnt = isoforms[0].size();
	set<int64> all_idx;
	for (unsigned i = 0; i < virtual_length.size(); i++)
		for_each_ele_in_group_const(iter, map_64_double, virtual_length[i])
			all_idx.insert(iter->first);

	int cnt = 0;
	map_64_32 map_to_flat_idx;
	for_each_ele_in_group(iter, set<int64>, all_idx)
		map_to_flat_idx[*iter] = cnt++;

	pair_triple_virtual_length.resize(isoforms.size());
	pair_triple_in_isoform.resize(isoforms.size());
	// Which pairs/triples are contained in an isoform
	for (unsigned i = 0; i < isoforms.size(); i++)
	{
		for_each_ele_in_group(iter, map_64_double, virtual_length[i])
		{
			if (all_idx.find(iter->first) != all_idx.end())
			{
				pair_triple_in_isoform[i].push_back(map_to_flat_idx[iter->first]);
				pair_triple_virtual_length[i].push_back(iter->second);
			}
		}
	}

	vector<int> start_exons;
	vector<int> end_exons;
	ExtractStartEndExons(isoforms, start_exons, end_exons);
	set<int> terminal_exons;
	for (unsigned i = 0; i < start_exons.size(); i++)
		terminal_exons.insert(start_exons[i]);
	for (unsigned i = 0; i < end_exons.size(); i++)
		terminal_exons.insert(end_exons[i]);

	vector<bool> b_bad_pts;
	b_bad_pts.assign(cnt, false);

	pair_triple_read.assign(cnt, 0);
	for_each_ele_in_group(iter, map_64_32, map_to_flat_idx)
	{
		int first_exon = -1;
		int second_exon = -1;
		int third_exon = -1;

		int64 idx = iter->first;
		int flat_idx = iter->second;
		if (idx < set_cnt * set_cnt)
		{
			first_exon = idx / set_cnt;
			second_exon = idx % set_cnt;
		}
		else
		{
			idx -= set_cnt * set_cnt;
			first_exon = idx / set_cnt / set_cnt;
			second_exon = (idx / set_cnt) % set_cnt;
			third_exon = idx % set_cnt;
		}

		vector<bool> a_part;
		a_part.assign(set_sizes.size(), false);
		a_part[first_exon] = true;
		a_part[second_exon] = true;
		if (third_exon >= 0) a_part[third_exon] = true;

		for (unsigned i = 0; i < pe_info.size(); i++)
		{
			for (unsigned j = 0; j < pe_info[i].mPartComb.size(); j++)
			{
				if (a_part == pe_info[i].mPartComb[j])
					pair_triple_read[flat_idx] += pe_info[i].mPartCombDup[j];
			}
		}

		if (!b_allpt)
		{
			if (terminal_exons.find(first_exon) != terminal_exons.end() ||
				terminal_exons.find(second_exon) != terminal_exons.end() ||
				terminal_exons.find(third_exon) != terminal_exons.end())
				b_bad_pts[flat_idx] = true;
		}
	}

	if (!b_allpt)
		RemoveMeasures(pair_triple_in_isoform, pair_triple_virtual_length, pair_triple_read, b_bad_pts);
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  VirtualLength
// Description:  For triples or pairs on each isoform, calculate the virtual length.
//               virtual_length  :  The index of a pair (i,j) is i * set_cnt + j;
//                                  The index of a triple (i,j,k) is 
//                                  i * set_cnt^2 + j * set_cnt + k + set_cnt^2
//                                  where set_cnt = isoforms[0].size();
//  Parameters:  
//--------------------------------------------------------------------------------------
double
IsoInferPE::VirtualLength(const vector<bool>& an_isoform, const vector<int>& set_sizes, 
						  const PEInfo& pe_info, map<int64, double>& virtual_length)
{
	int set_cnt = an_isoform.size();
	int read_len = pe_info.mReadLen;
	int max_gap = pe_info.mSpanHigh;

	// Extract all the existing paired partial combination of current isoform.
	for (int start = 0; start < set_cnt; start++)
	{
		if (!an_isoform[start]) continue;
		for (int end = start+1; end < set_cnt; end++)
		{
			if (!an_isoform[end]) continue;
			int gap = 0;
			for (int idx = start + 1; idx <= end - 1; idx++)
				if (an_isoform[idx]) gap += set_sizes[idx];

			int idx = start * set_cnt + end;
			double v_len = VirtualLength(pe_info, set_sizes[start]-read_len, gap + 2*read_len, set_sizes[end]-read_len);
			v_len *= (double)pe_info.mReadCnt / 1000000000.0;
			if (virtual_length.find(idx) == virtual_length.end())
				virtual_length[idx] = v_len;
			else
				virtual_length[idx] += v_len;
			
			if (gap >= max_gap) break;
		}
	}

	// Extract all the existing triple partial combination of current isoform.
	int pair_start = -1;
	while (true)
	{
		int last_one = pair_start;
		pair_start++;
		while (pair_start < set_cnt && !an_isoform[pair_start])
			pair_start++;
		if (pair_start == set_cnt) break;

		if (-1 == last_one) continue;

		for (int end = pair_start + 1; end < set_cnt; end++)
		{
			if (!an_isoform[end]) continue;
			int gap = set_sizes[pair_start];
			for (int idx = pair_start + 1; idx <= end - 1; idx++)
				if (an_isoform[idx]) gap += set_sizes[idx];

			// Make sure that last_one < pair_start < end
			int idx = last_one * set_cnt * set_cnt + pair_start * set_cnt + end + set_cnt * set_cnt;
			double v_len = VirtualLength(pe_info, read_len, gap+read_len, set_sizes[end]-read_len);
			v_len *= (double)pe_info.mReadCnt / 1000000000.0;
			if (virtual_length.find(idx) == virtual_length.end())
				virtual_length[idx] = v_len;
			else
				virtual_length[idx] += v_len;

			if (gap >= max_gap) break;
		}

		for (int end = last_one - 1; end >= 0; end--)
		{
			if (!an_isoform[end]) continue;
			int gap = set_sizes[last_one];
			for (int idx = end + 1; idx <= last_one - 1; idx++)
				if (an_isoform[idx]) gap += set_sizes[idx];

			// Make sure that end < last_one < pair_start
			int idx = end * set_cnt * set_cnt + last_one * set_cnt + pair_start + set_cnt * set_cnt;
			double v_len = VirtualLength(pe_info, read_len, gap+read_len, set_sizes[end]-read_len);
			v_len *= (double)pe_info.mReadCnt / 1000000000.0;
			if (virtual_length.find(idx) == virtual_length.end())
				virtual_length[idx] = v_len;
			else
				virtual_length[idx] += v_len;

			if (gap >= max_gap) break;
		}
	}
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  VirtualLength
// Description: 
//  Parameters:  
//      Return:  The returned value multiplied by expression level (in RPKM) is the 
//               expected #observed reads on this pair/triple
//--------------------------------------------------------------------------------------
double
IsoInferPE::VirtualLength(const PEInfo& pe_info, int len_first_interval,
						  int len_second_interval, int len_third_interval)
{
	double span_std = pe_info.mSpanStd;
	double span_mean = pe_info.mSpanMean;
	int read_len = pe_info.mReadLen;

	//    small exon                     large exon
	//     -------                      -------------
	//  ====     ====                ====           ====                        
	double prob_sum = 0;
	for (int i = 0; i < len_first_interval; i++)
	{
		int lower = i + len_second_interval;
		int upper = i + len_second_interval + len_third_interval;

		double prob_in_range = gsl_cdf_gaussian_P((upper - span_mean) / span_std, 1) -
							   gsl_cdf_gaussian_P((lower - span_mean) / span_std, 1);

		prob_sum += prob_in_range;
	}

	//cout << len_first_interval << " , " << len_second_interval << " , "
	//     << len_third_interval << " , " << prob_sum << endl;
	return prob_sum;
}

//--------------------------------------------------------------------------------------
//       Class:  IsoInferPE
//      Method:  BestCombination
// Description:  If the number of valid isoforms is small (less than 20), this method
//               enumerates all possible combinations and find the best one. This method
//               also needs the constructed set cover problem.
//  Parameters:  largest_ele : The largest element of the set cover instance.
//--------------------------------------------------------------------------------------
	void
IsoInferPE::BestCombination (   const vector<int>& set_sizes, 
								const vector<int>& exon_type, 
								const vector<double>& start_from_cnt, 
								const vector<double>& end_at_cnt, 
								const vector<vector<double> >& junc_cnt, 
								const vector<vector<bool> >& valid_isoforms,
								const vector<bool>& b_known_isoforms,
								const vector<vector<int> >& start_exons, 
								const vector<vector<int> >& end_exons, 
								const vector<PEInfo>& pe_info,
								bool b_enable_se,
								vector<int>& solution)
{
	vector<vector<int> > sc_sets;
	int largest_ele = ConstructSetCover(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, valid_isoforms, 
										start_exons, end_exons, pe_info, b_enable_se, sc_sets);

	if (valid_isoforms.size() > 20)
	{
		vector<double> sc_set_weight;
		sc_set_weight.assign(valid_isoforms.size(), 1);
		for (unsigned i = 0; i < valid_isoforms.size(); i++)
			if (b_known_isoforms[i]) sc_set_weight[i] = 0;

		SolveSetCover(largest_ele, sc_set_weight, sc_sets, solution);
		return;
	}

	//cout << "Find the best combination from " << valid_isoforms.size() << " isoforms." << endl;
	double best_obj = numeric_limits<double>::max();
	vector<int> best_solution;
	best_solution.resize(sc_sets.size());
	for (unsigned i = 0; i < best_solution.size(); i++)
		best_solution[i] = i;

	time_t start_time, end_time;
	time(&start_time);

	//mOmittedJunctionCnt = 0;
	mpSolver->SetParameters(mOmittedJunctionCnt / (mpInstance->mReadLen - mpInstance->mCrossStrength), 
							mpInstance->mReadLen, mpInstance->mCrossStrength);

	vector<vector<int> > measure_in_isoform;
	vector<vector<double> > measure_virtual_length;
	vector<double> measure_read;

	ConstructLPInstance(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, valid_isoforms, pe_info, 
						measure_in_isoform, measure_virtual_length, measure_read);

	vector<int> new_valid_isoforms_idx;
	vector<int> known_isoforms_idx;
	for (unsigned i = 0; i < valid_isoforms.size(); i++)
		if (b_known_isoforms[i]) known_isoforms_idx.push_back(i);
		else new_valid_isoforms_idx.push_back(i);

	vector<int> selected_new_idx;
	vector<bool> is_measure_covered;
	bool b_break = false;
	// Enumerate all possible combinations
	for (unsigned iso_cnt = 0; iso_cnt <= new_valid_isoforms_idx.size(); iso_cnt++)
	{
		selected_new_idx.resize(iso_cnt+1);
		for (int i = 0; i <= iso_cnt; i++)
			selected_new_idx[i] = i;
		do
		{
			solution.clear();
			for (unsigned i = 0; i < known_isoforms_idx.size(); i++)
				solution.push_back(known_isoforms_idx[i]);
			for (int i = 0; i < iso_cnt; i++)
				solution.push_back(new_valid_isoforms_idx[selected_new_idx[i]]);

			is_measure_covered.assign(largest_ele+1, false);
			for (unsigned i = 0; i < solution.size(); i++)
			{
				int curr_iso = solution[i];
				for (unsigned j = 0; j < sc_sets[curr_iso].size(); j++)
					is_measure_covered[sc_sets[curr_iso][j]] = true;
			}
			bool b_feasible = true;
			for (unsigned i = 0; i < is_measure_covered.size(); i++)
				if (!is_measure_covered[i]) b_feasible = false;

			// If current combination is feasible.
			if (b_feasible)
			{
				vector<vector<int> > selected_measure_in_isoform;
				vector<vector<double> > selected_measure_virtual_length;
				for (unsigned i = 0; i < iso_cnt; i++)
				{
					selected_measure_in_isoform.push_back(measure_in_isoform[solution[i]]);
					selected_measure_virtual_length.push_back(measure_virtual_length[solution[i]]);
				}

				double obj = mpSolver->SolveLeastSquareCon(selected_measure_in_isoform, selected_measure_virtual_length, measure_read);
				double pvalue = mpSolver->ResultPvalue();

				if (best_obj > obj)
				{
					best_obj = obj;
					best_solution = solution;
				}

				time(&end_time);
				double time_diff = difftime(end_time, start_time);
				if (time_diff > 200) 
				{
					b_break = true;
					break;
				}

				if (pvalue > 0.05)
					b_break = true;
			}

		} while (Utility2::NextCombination(selected_new_idx, measure_in_isoform.size()));

		if (b_break) break;
	}

	solution = best_solution;
}


