/*
 * =====================================================================================
 *
 *       Filename:  InstanceWriter.h
 *
 *    Description:  The header file for writing an instance
 *
 *        Version:  1.0
 *        Created:  04/09/2009 12:20:42 PM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */
#ifndef InstanceWriter_H
#define InstanceWriter_H

#include "InstanceHandler.h"

// For setw()
#include <iomanip>

/*
 * =====================================================================================
 *        Class:  InstanceWriter
 *  Description:  The class for writing an instance
 * =====================================================================================
 */
class InstanceWriter : public InstanceHandler
{
	public:
		// WRITE_HUI : Output the format needed by the matlab code of HuiJiang
		// WRITE_EXP : Output the expression levels of each isoform.
		enum WriteFormat {WRITE_FULL, WRITE_PREDICTION, WRITE_EXP, WRITE_EXPRPKM, WRITE_HUI, WRITE_TSSPAS, WRITE_EXON, WRITE_ISO};

		/* ====================  LIFECYCLE     ======================================= */
		InstanceWriter (ostream* p_output = NULL) : InstanceHandler(p_output) 
		{
			mFormat = WRITE_FULL;
		};
		/* constructor      */

		~InstanceWriter (){};                            /* destructor       */

		void
		SetConfidencelevel(double confi_level){mConfidenceLevel = confi_level;}

		void
		SetWriteFormat(WriteFormat format)
		{
			mFormat = format;
		}

		virtual
		void
		Initialize()
		{
			bHuiHead = true;
		}

		virtual
		void
		CleanUp()
		{
		}

		/* ====================  OPERATORS     ======================================= */
		virtual
		void
		OnInstance (Instance& an_instance)
		{
			switch (mFormat)
			{
			case WRITE_FULL:
				WriteAll(an_instance);
				break;
			case WRITE_PREDICTION:
				WritePrediction(an_instance);
				break;
			case WRITE_EXP:
				WriteExp(an_instance);
				break;
			case WRITE_EXPRPKM:
				WriteExpRPKM(an_instance);
				break;
			case WRITE_HUI:
				WriteHui(an_instance);
				break;
			case WRITE_TSSPAS:
				WriteTSSPAS(an_instance);
				break;
			case WRITE_EXON:
				WriteExon(an_instance);
				break;
			case WRITE_ISO:
				WriteIso(an_instance);
				break;
			default:
				WriteAll(an_instance);
			}
			InstanceHandler::OnInstance(an_instance);
		}
	protected:

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteExp
		 *  Description:  This methods writes expression of each isoform.
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteExp(Instance& an_instance)
		{
			string mark = "New";
			if (an_instance.mPvalue >= mConfidenceLevel && an_instance.mIsoforms.size() == an_instance.mKnownCnt)
				mark = "No";
			
			for (unsigned i = 0; i < an_instance.mIsoExp.size(); i++)
				(*mpOutput) << an_instance.mGenes[i].mName << "\t" << an_instance.mIsoExp[i] << "\t" << mark << "\t" << an_instance.mInstanceCnt << endl;
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteExpRPKM
		 *  Description:  This methods writes expression of each isoform.
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteExpRPKM(Instance& an_instance)
		{
			string mark = "New";
			if (an_instance.mPvalue >= mConfidenceLevel && an_instance.mIsoforms.size() == an_instance.mKnownCnt)
				mark = "No";
			
			double ratio = 1000000000.0 / ((double)an_instance.mReadCnt);
			for (unsigned i = 0; i < an_instance.mIsoExp.size(); i++)
				(*mpOutput) << an_instance.mGenes[i].mName << "\t" << an_instance.mIsoExp[i] * ratio << "\t" << mark << "\t" << an_instance.mInstanceCnt << endl;
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WritePrediction
		 *  Description:  This methods writes the predictions
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WritePrediction(Instance& an_instance)
		{
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteAll
		 *  Description:  This methods writes all the information of a read. The format should
		 *                be recoganized by a reader
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteAll(Instance& an_instance)
		{
			int 					&instance_cnt =     an_instance.mInstanceCnt;
			vector<int> 			&set_sizes =        an_instance.mSetSizes;
			vector<double> 			&sample_cnt =       an_instance.mSampleCnt;
			vector<vector<bool> > 	&isoforms =         an_instance.mIsoforms;
			vector<double> 			&iso_exp =          an_instance.mIsoExp;
			vector<vector<double> >	&splice_read_cnt =  an_instance.mSpliceReadCnt;
			vector<vector<double> >	&splice_read_cnt_pe =  an_instance.mSpliceReadCntPE;
			vector<Exon> 			&exons =            an_instance.mExons;
			vector<Gene> 			&genes =        	an_instance.mGenes;
			string 					&comments =         an_instance.mComments;
			vector<double>          &fit_value =        an_instance.mFitValue;
			double                  &noise_level =      an_instance.mNoiseLevel;
			int                     &known_cnt  =       an_instance.mKnownCnt;
			int                     &read_len   =       an_instance.mReadLen;
			int                     &read_cnt   =       an_instance.mReadCnt;
			int                     &cross_strengh  =   an_instance.mCrossStrength;
			vector<PEInfo>          &all_pe_info =      an_instance.mPEInfo;

			(*mpOutput) << "Instance\t" << instance_cnt << "\t" << set_sizes.size() << "\t" << isoforms.size() << "\t" << known_cnt << "\t" << comments << endl;
			(*mpOutput) << "read len  :\t" << read_len << endl;
			(*mpOutput) << "read cnt  :\t" << read_cnt << endl;
			(*mpOutput) << "cross str :\t" << cross_strengh << endl;
			(*mpOutput) << "noise lev :\t" << noise_level << endl;

			int set_cnt = set_sizes.size();
			int iso_cnt = isoforms.size();
			if (exons.size() == set_sizes.size())
			{
				(*mpOutput) << "genes     :\t" << genes.size() << endl;
				for (int i = 0; i < genes.size(); i++)
					genes[i].Write(mpOutput);

				(*mpOutput) << "exp segs  :" << endl;
				for (int i = 0; i < set_cnt; i++)
					exons[i].Write(mpOutput);
			}

			if (fit_value.size() != 0)
			{
				(*mpOutput) << "fit value :";
				for (int i = 0; i < set_cnt; i++)
					(*mpOutput) << "\t" << setw(6) << setprecision(3) << fit_value[i];
				(*mpOutput) << endl;
			}
			(*mpOutput) << "sample cnt:";
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << "\t" << setw(6) << sample_cnt[i];
			(*mpOutput) << endl;

			(*mpOutput) << "set size  :";
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << "\t" << setw(6) << set_sizes[i];
			(*mpOutput) << endl;

			for (int i = 0; i < iso_cnt; i++)
			{
				if (i < known_cnt)
					(*mpOutput) << "iso  " << setw(4) << i;
				else
					(*mpOutput) << "N iso" << setw(4) << i;
				for (int j = 0; j < set_cnt; j++)
					(*mpOutput) << "\t" << setw(6) << isoforms[i][j];
				(*mpOutput) << "\t" << setw(6) << iso_exp[i] << endl;
			}

			// Output junc reads
			for (unsigned i = 0; i < splice_read_cnt.size(); i++)
			{
				(*mpOutput) << "           ";
				for (unsigned j = 0; j < splice_read_cnt[i].size(); j++)
					(*mpOutput) << "\t" << setw(6) << splice_read_cnt[i][j];
				(*mpOutput) << endl;
			}

			// Output PE information
			for (unsigned i = 0; i < all_pe_info.size(); i++)
			{
				PEInfo& pe_info = all_pe_info[i];
				(*mpOutput) << "PE info:\t" << pe_info.mReadLen << "\t" << pe_info.mSpanMean << "\t" <<
						   pe_info.mSpanStd << "\t" << pe_info.mReadCnt << "\t" <<
						   pe_info.mPartComb.size() << endl;
				for (unsigned j = 0; j < pe_info.mPartComb.size(); j++)
				{
					for (unsigned k = 0; k < pe_info.mPartComb[j].size(); k++)
						(*mpOutput) << pe_info.mPartComb[j][k] << "\t";
					(*mpOutput) << pe_info.mPartCombDup[j] << endl;
				}
			}

			if (all_pe_info.size() > 0)
			{
				// Output junc reads obtained by PE reads
				for (unsigned i = 0; i < splice_read_cnt_pe.size(); i++)
				{
					for (unsigned j = 0; j < splice_read_cnt_pe[i].size(); j++)
						(*mpOutput) << "\t" << setw(6) << splice_read_cnt_pe[i][j];
					(*mpOutput) << endl;
				}
			}

			// Output start end exons
			(*mpOutput) << "SE exons:\t" << an_instance.mStartExons.size() << endl;
			for (unsigned i = 0; i < an_instance.mStartExons.size(); i++)
			{
				(*mpOutput) << an_instance.mStartExons[i][0];
				for (unsigned j = 1; j < an_instance.mStartExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mStartExons[i][j];
				(*mpOutput) << "\t";
				(*mpOutput) << an_instance.mEndExons[i][0];
				for (unsigned j = 1; j < an_instance.mEndExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mEndExons[i][j];
				(*mpOutput) << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteHui
		 *  Description:  This methods writes the instance needed by the matlab code of HuiJiang 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteHui(Instance& an_instance)
		{
			int 					&instance_cnt =     an_instance.mInstanceCnt;
			vector<int> 			&set_sizes =        an_instance.mSetSizes;
			vector<double> 			&sample_cnt =       an_instance.mSampleCnt;
			vector<vector<bool> > 	&isoforms =         an_instance.mIsoforms;
			vector<vector<double> > &splice_read_cnt =  an_instance.mSpliceReadCnt;
			vector<Gene> 			&genes =            an_instance.mGenes;
			int                     &read_len   =       an_instance.mReadLen;

			int set_cnt = set_sizes.size();
			int iso_cnt = isoforms.size();

			if (set_cnt > 40) return;

			if (bHuiHead)
			{
				(*mpOutput) << an_instance.mReadCnt << "\t" << an_instance.mReadLen << endl;
				bHuiHead = false;
			}

			(*mpOutput) << instance_cnt << "\t" << set_cnt << "\t" << iso_cnt << endl;
			for (int i = 0; i < iso_cnt; i++)
				(*mpOutput) << genes[i].mName << "\t";
			(*mpOutput) << endl;

			// set size
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << set_sizes[i] << "\t";
			(*mpOutput) << endl;

			// sample cnt
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << sample_cnt[i] << "\t";
			(*mpOutput) << endl;

			// junc length 
			for (int i = 0; i < set_cnt; i++)
				for (int j = i+1; j < set_cnt; j++)
					(*mpOutput) << 2*read_len-1 << "\t";
			(*mpOutput) << endl;

			// junc reads
			for (int i = 0; i < set_cnt; i++)
				for (int j = i+1; j < set_cnt; j++)
					(*mpOutput) << splice_read_cnt[i][j] << "\t";
			(*mpOutput) << endl;

			// iso
			for (int i = 0; i < iso_cnt; i++)
			{
				for (int j = 0; j < set_cnt; j++)
					(*mpOutput) << isoforms[i][j] << "\t";
				(*mpOutput) << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteTSSPAS
		 *  Description:  This methods writes the TSSs and PASs of each instance 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteTSSPAS(Instance& an_instance)
		{
			(*mpOutput) << an_instance.mInstanceCnt << "\t"
			            << an_instance.mExons[0].mChr << "\t"
			            << (an_instance.mExons[0].mStrand ? '+' : '-') << "\t"
				        << an_instance.mExons[0].mStart << "\t"
				        << an_instance.mExons[an_instance.mExons.size()-1].mEnd << "\t"
				        << "RANGE" << endl;
			for (unsigned i = 0; i < an_instance.mStartExons.size(); i++)
			{
				(*mpOutput) << an_instance.mInstanceCnt << "\t"
							<< an_instance.mExons[an_instance.mStartExons[i][0]].mStart;
				for (unsigned j = 1; j < an_instance.mStartExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mExons[an_instance.mStartExons[i][j]].mStart;

				(*mpOutput) << "\t" << an_instance.mExons[an_instance.mEndExons[i][0]].mEnd;
				for (unsigned j = 1; j < an_instance.mEndExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mExons[an_instance.mEndExons[i][j]].mEnd;
				(*mpOutput) << "\t" << "TSSPAS" << endl;
			}
			for (unsigned i = 0; i < an_instance.mExons.size(); i++)
			{
				(*mpOutput) << an_instance.mExons[i].mChr << "\t"
							<< (an_instance.mExons[i].mStrand ? '+' : '-') << "\t"
							<< an_instance.mExons[i].mStart << "\tBOUND" << endl;
				(*mpOutput) << an_instance.mExons[i].mChr << "\t"
							<< (an_instance.mExons[i].mStrand ? '+' : '-') << "\t"
							<< an_instance.mExons[i].mEnd << "\tBOUND" << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteIso
		 *  Description:  This methods writes the TSSs and PASs of each instance 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteIso(Instance& an_instance)
		{
			vector<Exon>& exons = an_instance.mExons;
			for (unsigned i = 0; i < an_instance.mIsoforms.size(); i++)
			{
				(*mpOutput) << an_instance.mGenes[i].mName << "\t" << exons[0].mChr << "\t" << (exons[0].mStrand ? '+' : '-') << "\t";
				
				const vector<bool>& an_iso = an_instance.mIsoforms[i];

				int first_idx = 0;
				for (first_idx = 0; first_idx < an_iso.size(); first_idx++)
					if (an_iso[first_idx])
						break;
				int last_idx = 0;
				for (last_idx = an_iso.size() - 1; last_idx >= 0; last_idx--)
					if (an_iso[last_idx])
						break;

				(*mpOutput) << exons[first_idx].mStart << "\t" << exons[last_idx].mEnd << "\t";

				bool b_first = true;
				for (unsigned j = 0; j < an_iso.size(); j++)
				{
					if (an_iso[j])
					{
						if (b_first) (*mpOutput) << exons[j].mStart;
						else (*mpOutput) << "," << exons[j].mStart;
						b_first = false;
					}
				}
				(*mpOutput) << "\t";

				b_first = true;
				for (unsigned j = 0; j < an_iso.size(); j++)
				{
					if (an_iso[j])
					{
						if (b_first) (*mpOutput) << exons[j].mEnd;
						else (*mpOutput) << "," << exons[j].mEnd;
						b_first = false;
					}
				}
				(*mpOutput) << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteExon
		 *  Description:  This methods writes the TSSs and PASs of each instance 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteExon(Instance& an_instance)
		{
			for (unsigned i = 0; i < an_instance.mExons.size(); i++)
			{
				(*mpOutput) << an_instance.mExons[i].mChr << "\t"
				            << an_instance.mExons[i].mStrand << "\t"
							<< an_instance.mExons[i].mStart << "\t"
				            << an_instance.mExons[i].mEnd << "\t"
				            << an_instance.mSampleCnt[i] << endl;
			}
		}


	private:
		double mConfidenceLevel;
		WriteFormat mFormat;
		bool bHuiHead;
}; /* -----  end of class InstanceWriter  ----- */

#endif
