// =====================================================================================
// 
//       Filename:  ExpressionCalculator.h
// 
//    Description:  This is the header file for class ExpressionCalculator
// 
//        Version:  1.0
//        Created:  06/11/2009 04:53:39 PM
//       Revision:  none
//       Compiler:  g++
// 
//         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
//        Company:  THU
// 
// =====================================================================================

#ifndef ExpressionCalculator_H 
#define ExpressionCalculator_H

#include <string>
#include <iostream>
#include "InstanceHandler.h"

using namespace std;

/*
 * =====================================================================================
 *        Class:  ExpressionCalculator
 *  Description:  This class calculate the expression levels of different isoforms 
 *                according to the mapped short reads
 * =====================================================================================
 */
class ExpressionCalculator : public InstanceHandler
{
	public:
		enum ExpMethod{EXP_METHOD_UNIQ, EXP_METHOD_QUAD, EXP_METHOD_LP};

		/* ====================  LIFECYCLE     ======================================= */
		ExpressionCalculator (LPsolver* p_solver, ostream* p_output = NULL) : InstanceHandler(p_output)
		{
			mpSolver = p_solver;
		};                             /* constructor */

		virtual ~ExpressionCalculator (){};                             /* constructor */

		void
		SetMethod(ExpMethod method){mMethod = method;}

		virtual
		void
		Initialize()
		{
		};

		virtual
		void
		OnInstance(Instance& an_instance)
		{
			switch (mMethod)
			{
			case EXP_METHOD_UNIQ:
				ExpOnUniqExonOrJunc(an_instance);
				break;
			case EXP_METHOD_QUAD:
				ExpOnQuadProg(an_instance);
				break;
			case EXP_METHOD_LP:
				ExpOnQuadProg(an_instance);
				break;
			default:
				ExpOnQuadProg(an_instance);
			}
			InstanceHandler::OnInstance(an_instance);
		}/* -----  end of method OnInstance  ----- */

		//--------------------------------------------------------------------------------------
		//       Class:  ExpressionCalculator
		//      Method:  ExpOnQuadProg
		// Description:  Calculate the expression levels with quadratic programming.
		//  Parameters:
		//--------------------------------------------------------------------------------------
		void
		ExpOnQuadProg(Instance& an_instance)
		{
			int 					&instance_cnt =     an_instance.mInstanceCnt;
			vector<int> 			&set_sizes =        an_instance.mSetSizes;
			vector<vector<bool> > 	&isoforms =         an_instance.mIsoforms;
			vector<vector<double> > &junc_cnt =         an_instance.mSpliceReadCnt;
			vector<Exon> 			&exons =            an_instance.mExons;
			vector<Gene> 			&genes =   		    an_instance.mGenes;
			vector<vector<int> >    &start_exons =      an_instance.mStartExons;
			vector<vector<int> >    &end_exons =        an_instance.mEndExons;
			vector<PEInfo>          &pe_info =          an_instance.mPEInfo;
			vector<double>          &iso_exp =          an_instance.mIsoExp;

			vector<double>          start_from_cnt;
			vector<double>          end_at_cnt;

			start_from_cnt.resize(exons.size());
			end_at_cnt.resize(exons.size());
			for (unsigned i = 0; i < exons.size(); i++)
			{
				start_from_cnt[i] = exons[i].mStartCnt;
				end_at_cnt[i] = exons[i].mEndCnt;
			}

			vector<vector<int> > measure_in_isoform;
			vector<vector<double> > measure_virtual_length;
			vector<double> measure_read;
			vector<int> exon_type;

			IsoInferPE isoinfer(mpSolver);
			isoinfer.SetInstance(&an_instance);
			isoinfer.CalculateExonType(start_exons, end_exons, set_sizes.size(), exon_type);
			isoinfer.ConstructLPInstance(set_sizes, exon_type, start_from_cnt, end_at_cnt, junc_cnt, isoforms, pe_info, 
										 measure_in_isoform, measure_virtual_length, measure_read);
			double obj_val = mpSolver->SolveLeastSquareCon(measure_in_isoform, measure_virtual_length, measure_read);
			double pvalue = mpSolver->ResultPvalue();
			mpSolver->GetExpLevel(iso_exp);
		}


		//--------------------------------------------------------------------------------------
		//       Class:  ExpressionCalculator
		//      Method:  ExpOnUniqExonOrJunc
		// Description:  Calculate the expression levels based on the unique exon or junctions
		//               of each isoform. Note that this function calculates the absolute 
		//               expression levels.
		//  Parameters:
		//--------------------------------------------------------------------------------------
		void
		ExpOnUniqExonOrJunc(Instance& an_instance)
		{
			vector<int> 			&set_sizes =        an_instance.mSetSizes;
			vector<double> 			&sample_cnt =       an_instance.mSampleCnt;
			vector<vector<bool> > 	&isoforms =         an_instance.mIsoforms;
			vector<vector<double> >	&splice_read_cnt =  an_instance.mSpliceReadCnt;
			vector<double> 			&iso_exp =          an_instance.mIsoExp;

			for (unsigned i = 0; i < set_sizes.size(); i++)
			{
				set_sizes[i] -= an_instance.mReadLen - 1;
				if (set_sizes[i] < 0) set_sizes[i] = 0;
			}

			// Decide which exons or junctions are unique
			vector<int> exon_visit_cnt;
			exon_visit_cnt.assign(set_sizes.size(), 0);
			for (unsigned i = 0; i < isoforms.size(); i++)
				for (unsigned j = 0; j < set_sizes.size(); j++)
					if (isoforms[i][j]) exon_visit_cnt[j]++;

			vector<vector<int> > junc_visit_cnt;
			junc_visit_cnt.resize(set_sizes.size());
			for (unsigned i = 0; i < set_sizes.size(); i++)
				junc_visit_cnt[i].assign(set_sizes.size(), 0);
			for (unsigned i = 0; i < isoforms.size(); i++)
			{
				int start = -1;
				for (unsigned j = 0; j < isoforms[i].size(); j++)
				{
					if (isoforms[i][j])
					{
						if (-1 == start)
							start = j;
						else 
						{
							junc_visit_cnt[start][j]++;
							start = j;
						}
					}
				}
			}

			iso_exp.resize(isoforms.size());
			// For each isoform, find the unique exon and isoforms
			for (unsigned i = 0; i < isoforms.size(); i++)
			{
				vector<int> uniq_exon;
				vector<int> uniq_junc_first;
				vector<int> uniq_junc_second;

				for (unsigned j = 0; j < set_sizes.size(); j++)
					if (isoforms[i][j] && exon_visit_cnt[j] == 1) uniq_exon.push_back(j);

				int start = -1;
				for (unsigned j = 0; j < isoforms[i].size(); j++)
				{
					if (isoforms[i][j])
					{
						if (-1 == start)
							start = j;
						else 
						{
							if (junc_visit_cnt[start][j] == 1)
							{
								uniq_junc_first.push_back(start);
								uniq_junc_second.push_back(j);
							}
							start = j;
						}
					}
				}

				// According to the unique exons or junctions, calculate the 
				// expression levels of each isoform.
				double len_sum = 0;
				double read_sum = 0;
				for (unsigned j = 0; j < uniq_junc_first.size(); j++)
					read_sum += splice_read_cnt[uniq_junc_first[j]][uniq_junc_second[j]];
				len_sum = uniq_junc_second.size() * (an_instance.mReadLen - an_instance.mCrossStrength);

				for (unsigned j = 0; j < uniq_exon.size(); j++)
				{
					read_sum += sample_cnt[uniq_exon[j]];
					len_sum += set_sizes[uniq_exon[j]] - an_instance.mReadLen + 1;
				}

				double exp = 0;
				if (len_sum != 0)
					exp = read_sum / len_sum;

				iso_exp[i] = exp;
			}

		}

		virtual
		void
		CleanUp(){}

	protected:

	private:

		ExpMethod mMethod;
}; /* -----  end of class ExpressionCalculator  ----- */

#endif
