/*
 * =====================================================================================
 *
 *       Filename:  Exon.h
 *
 *    Description:  The header file for class Exon
 *
 *        Version:  1.0
 *        Created:  04/09/2009 12:15:02 PM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef Exon_H 
#define Exon_H

#include <vector>
#include <ostream>
#include <string>
#include <map>
#include "Utility2.h"
#include <limits>

using namespace std;

class Segment
{
public:
	string mChr;
	//[mStart, mEnd) is the segment
	int	mStart;
	int mEnd; 
	bool mStrand; // true +; false -
	string mSeq;

	Segment()
	{
		mStrand = true;
	}

	bool operator< (const Segment& seg) const
	{
		if (mStart < seg.mStart)
			return true;
		if (mStart > seg.mStart)
			return false;
		if (mEnd < seg.mEnd)
			return true;
		if (mEnd > seg.mEnd)
			return false;
		return false;
	}

	bool operator != (const Segment& seg) const
	{
		return (mStart != seg.mStart || mEnd != seg.mEnd);
	}

	bool operator == (const Segment& seg) const
	{
		return (mStart == seg.mStart && mEnd == seg.mEnd);
	}

	virtual void Write(ostream* p_out = NULL) const
	{
		if (!p_out)
			p_out = &cout;
		(*p_out) << mChr << "\t" 
			 << (mStrand ? '+' : '-' ) << "\t" 
			 << mStart << "\t" 
		     << mEnd << endl;
	}
};

class Exon : public Segment
{
public:
	using Segment::operator ==;
	using Segment::operator !=;
	using Segment::operator <;

	string mGene;        // Which gene is this exon belonging to
	int mBothCnt; 	     // The number of reads whose start and end points are in this exon
	int mStartCnt;       // The number of reads whose start point is in this exon
	int mEndCnt;         // The number of reads whose end point is in this exon
	
	int mStartType;      // refers to Boundary
	int mEndType;        // refers to Boundary

	Exon()
	{
		mBothCnt = 0;
		mStartCnt = 0;
		mEndCnt = 0;
	}

	/* virtual */
	void Write(ostream* p_out = NULL) const
	{
		if (!p_out)
			p_out = &cout;
		(*p_out) << mChr << "\t" 
			 << (mStrand ? '+' : '-' ) << "\t" 
			 << mStart << "\t" 
		     << mEnd << "\t" 
			 << mStartCnt << "\t"
			 << mEndCnt << "\t"
			 << mBothCnt << "\t"
			 << mStartType << "\t"
			 << mEndType << endl;
	}

	void Read(string& line)
	{
		vector<string> splitted = Utility::Split('\t', line);
		mChr = splitted[0];
		mStrand = (splitted[1] == "+");
		mStart = atoi(splitted[2].data());
		mEnd = atoi(splitted[3].data());
		mStartCnt = atoi(splitted[4].data());
		mEndCnt = atoi(splitted[5].data());
		mBothCnt = atoi(splitted[6].data());
		mStartType = atoi(splitted[7].data());
		mEndType = atoi(splitted[8].data());
	}
};

class Gene : public Segment
{
public:
	using Segment::operator ==;
	using Segment::operator !=;
	using Segment::operator <;

	string mName;
	vector<Exon> mExons;
	// Isoforms start from an element in mTSSs[i] must end with some element in mPASs[i]
	vector<vector<int64> > mTSSs;
	vector<vector<int64> > mPASs;

	Gene()
	{
		
	}

	// Calculate the range of this gene
	void CalculateRange()
	{
		mStart = numeric_limits<int64>::max();
		mEnd = 0;
		for (unsigned i = 0; i < mExons.size(); i++)
		{
			if (mStart > mExons[i].mStart) mStart = mExons[i].mStart;
			if (mEnd < mExons[i].mEnd) mEnd = mExons[i].mEnd;
		}
	}

	/* virtual */
	void Write(ostream* p_out = NULL) const
	{
		if (!p_out)
			p_out = &cout;
		(*p_out) << mName << "\t" 
		     << mChr << "\t" 
			 << (mStrand ? '+' : '-') << "\t" 
			 << mStart << "\t" 
			 << mEnd;

		(*p_out) << "\t" << mExons[0].mStart;
		for (unsigned i = 1; i < mExons.size(); i++)
			(*p_out) << "," << mExons[i].mStart;

		(*p_out) << "\t" << mExons[0].mEnd;
		for (unsigned i = 1; i < mExons.size(); i++)
			(*p_out) << "," << mExons[i].mEnd;
		(*p_out) << endl;
	}

	void Read(string& line)
	{
		vector<string> splitted = Utility::Split('\t', line);
		mName = splitted[0];
		mChr = splitted[1];
		mStrand = (splitted[2] == "+");

		mStart = atoi(splitted[3].data());
		mEnd = atoi(splitted[4].data());
		
		vector<string> starts = Utility::Split(',', splitted[5]);
		mExons.resize(starts.size());
		for (unsigned i = 0; i < starts.size(); i++)
			mExons[i].mStart = atoi(starts[i].data());

		vector<string> ends = Utility::Split(',', splitted[6]);
		for (unsigned i = 0; i < ends.size(); i++)
			mExons[i].mEnd = atoi(ends[i].data());
	}

};

class Boundary
{
public:
	int64 mPos;   // The position of the boundary
	int mType;  // 0  :  intron->exon
	            // 1  :  exon->intron
				// 2  :  intron->exon and exon->intron

	bool operator< (const Boundary& bound) const
	{
		if (mPos < bound.mPos)
			return true;
		return false;
	}

	bool operator != (const Boundary& bound) const
	{
		return (mPos != bound.mPos);
	}

	bool operator == (const Boundary& bound) const
	{
		return (mPos == bound.mPos);
	}

	virtual void Write(ostream* p_out = NULL) const
	{
		if (!p_out)
			p_out = &cout;
		(*p_out) << mPos << "\t" << mType << endl;
	}
};

typedef map<string, vector<Segment> > map_str2vec_seg;
typedef map<int64, int> map_int642int;
typedef map<string, set<int> > map_str2set_int;
typedef map<string, vector<int> > map_str2vec_int;
typedef map<string, vector<Exon> > map_str2vec_exon;
typedef map<string, vector<Gene> > map_str2vec_gene;
typedef map<string, vector<Boundary> > map_str2vec_boundary;
typedef map<string, map<int64, int> > map_str2map_int642int;

#endif
