/*
 * =====================================================================================
 *
 *       Filename:  GraphAlgorithm_MaxFlow.hpp
 *
 *    Description:  An implementation of maxflow algorithm
 *
 *        Version:  1.0
 *        Created:  05/26/2008 10:00:23 AM
 *       Modified:  10/27/2008
 *       Revision:  none
 *       Compiler:  g++
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef GraphAlgorithm_MaxFlow_H
#define GraphAlgorithm_MaxFlow_H

#include "Graph.h"

/*
 * =====================================================================================
 *        Class:  GraphAlgorithm_MaxFlow
 *  Description:  Provides algorithms to calculate the maxflow of a network
 * =====================================================================================
 */
template < typename FLOW_TYPE >
class GraphAlgorithm_MaxFlow
{
	public:

		/* ====================  LIFECYCLE     ======================================= */
		GraphAlgorithm_MaxFlow ();                             /* constructor      */
		~GraphAlgorithm_MaxFlow ();                            /* destructor       */

		void SetGraph (Graph* pGraph, int source, int sink, FLOW_TYPE* capacity);
		void SetErrorTolerance(FLOW_TYPE tolerance){mErrorTolerance = tolerance;}
		void PrepareBuffers();
		void ClearFlow();
		FLOW_TYPE MaxFlow();
		FLOW_TYPE Flow(int edge){return mpFlow[edge];}

		void MinCut(int* nodes, bool b_min_source);

		void ClearBuffers();


	protected:

	private:
		void NullizePointers();
		bool Relabel(int node);
		void GapRelabel (int node, int old_label);
		void GapRelabelInit();

		void DFSResidualFromSource(bool* visited, int curr_node);
		void DFSResidualFromSink(bool* visited, int curr_node);

		Graph* 				mpGraph;
		int         		mSource;
		int         		mSink;
		FLOW_TYPE 			mErrorTolerance;
		FLOW_TYPE*  	  	mpCapacity;
		FLOW_TYPE*    		mpFlow;
		int*                mpLabel;
		FLOW_TYPE*    		mpExcess;
		bool*               mpIsExcessed;
		int*                mpExcessedNode;
		int*                mpNextActiveEdgeIndex;
		int*                mpFirstNodeWithCertainLabel;
		int*                mpNextNodeWithCertainLabel;
		int*                mpPrevNodeWithCertainLabel;
		int**               mpAdjEdges;

}; /* -----  end of class  GraphAlgorithm_MaxFlow  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  GraphAlgorithm_MaxFlow
 * Description:  constructor
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
GraphAlgorithm_MaxFlow<FLOW_TYPE>::GraphAlgorithm_MaxFlow ()
{
	mSource = -1;
	mSink = -1;
	mErrorTolerance = 0;
	NullizePointers();
}  /* -----  end of method GraphAlgorithm_MaxFlow::GraphAlgorithm_MaxFlow  (constructor)  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  NullizePointers
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::NullizePointers (  )
{
	mpGraph = 0;
	mpCapacity = 0;
	mpFlow = 0;
	mpLabel = 0;
	mpExcess = 0;
	mpExcessedNode = 0;
	mpIsExcessed = 0;
	mpNextActiveEdgeIndex = 0;
	mpFirstNodeWithCertainLabel = 0;
	mpNextNodeWithCertainLabel = 0;
	mpPrevNodeWithCertainLabel = 0;
	mpAdjEdges = 0;
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::NullizePointers  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  ~GraphAlgorithm_MaxFlow
 * Description:  destructor
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
GraphAlgorithm_MaxFlow<FLOW_TYPE>::~GraphAlgorithm_MaxFlow ()
{
}  /* -----  end of method GraphAlgorithm_MaxFlow::~GraphAlgorithm_MaxFlow  (destructor)  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  SetGraph
 * Description:  
 *       Param:  pGraph     :     The graph on which the algorithm operates
 *               source     :     The inner ID of the souce node in the graph
 *               sink       :     The inner ID of the sink node in the graph
 *               capacity   :     The capacity of every edge. This array is indexed by
 *                                the inner ID of the edges. During the execution of the
 *                                algorithm, the space for this array should not be altered
 *                                or deleted.
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::SetGraph (Graph* pGraph, int source, int sink, FLOW_TYPE* capacity)
{
	mpGraph = pGraph;
	mSource = source;
	mSink = sink;
	mpCapacity = capacity;
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::SetGraph  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  PrepareBuffers
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::PrepareBuffers (  )
{
	if (mpFlow)
		return;
	int all_node_cnt = mpGraph->NodeCnt(false);
	int all_edge_cnt = mpGraph->EdgeCnt(false);
	int unmasked_node_cnt = mpGraph->NodeCnt();
	int unmasked_edge_cnt = mpGraph->EdgeCnt();

	mpFlow = new FLOW_TYPE[all_edge_cnt];
	const int* edges = mpGraph->Edges();
	for (int i = 0; i < unmasked_edge_cnt; i++)
	{
		int edge = edges[i];
		mpFlow[edge] = 0;
	}

	mpExcess = new FLOW_TYPE[all_node_cnt];
	mpLabel = new int[all_node_cnt];
	mpExcessedNode = new int[all_node_cnt];
	mpIsExcessed = new bool[all_node_cnt];
	mpNextActiveEdgeIndex = new int[all_node_cnt];
	mpFirstNodeWithCertainLabel = new int[all_node_cnt];
	mpNextNodeWithCertainLabel = new int[all_node_cnt];
	mpPrevNodeWithCertainLabel = new int[all_node_cnt];
	mpAdjEdges = new int*[all_node_cnt];
	bool b_mask = mpGraph->EnableMask(false);
	for (int i = 0; i < all_node_cnt; i++)
		mpAdjEdges[i] = new int[mpGraph->IoDegree(i)];
	mpGraph->EnableMask(b_mask);

	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::PrepareBuffers  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  ClearBuffers
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::ClearBuffers (  )
{
	if (!mpFlow)
		return;
	delete[] mpFlow;
	delete[] mpExcess;
	delete[] mpLabel;
	delete[] mpIsExcessed;
	delete[] mpNextActiveEdgeIndex;
	delete[] mpFirstNodeWithCertainLabel;
	delete[] mpNextNodeWithCertainLabel;
	delete[] mpPrevNodeWithCertainLabel;

	int all_node_cnt = mpGraph->NodeCnt(false);
	for (int i = 0; i < all_node_cnt; i++)
		delete[] mpAdjEdges[i];
	delete[] mpAdjEdges;

	NullizePointers();	
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::ClearBuffers  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  Relabel
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	bool	
GraphAlgorithm_MaxFlow<FLOW_TYPE>::Relabel ( int node )
{
	// For error in the floating calculation. a relabel operation may fail.
	// i.e. There is overflow(very small) on some node, but it can not push it to its 
	// neighbors because every edge out of it to a neighbor with small mpLabel is full
	// The mErrorTolerance can not handle this because, in difference situations, this error
	// is different. if mErrorTolerance is too large the maxflow may be not optimal. Although
	// it seems no big deal if only the flow value is needed, the densest subgraph
	// is sensitive to this.

	bool b_succ = false;
	int unmasked_node_cnt = mpGraph->NodeCnt();
	// Relabel
	int min_height = unmasked_node_cnt*2+2;
	const int* edges = mpGraph->OutEdges(node);
	for (int i = 0; i < mpGraph->OutDegree(node); i++)
	{
		int edge = edges[i];
		// If pEdge is not admissible
		if (mpCapacity[edge] - mpFlow[edge] <= mErrorTolerance)
			continue;
		int to_node = mpGraph->ToNode(edge);
		if (min_height > mpLabel[to_node])
		{
			min_height = mpLabel[to_node];
			b_succ = true;
		}
	}
	// Residual edges
	edges = mpGraph->InEdges(node);
	for (int i = 0; i < mpGraph->InDegree(node); i++)
	{
		int edge = edges[i];
		// If pEdge is not admissible
		if (mpFlow[edge] <= mErrorTolerance)
			continue;
		int from_node = mpGraph->FromNode(edge);
		if (min_height > mpLabel[from_node])
		{
			min_height = mpLabel[from_node];
			b_succ = true;
		}
	}
	mpLabel[node] = min_height + 1;
	return b_succ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::Relabel  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  GapRelabelInit
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::GapRelabelInit (  )
{
	// Initialize mpFirstNodeWithCertainLabel, mpNextNodeWithCertainLabel and mpPrevNodeWithCertainLabel for Gap relabeling
	int unmasked_node_cnt = mpGraph->NodeCnt();
	if (unmasked_node_cnt == 0)
	{
		cerr << "GapRelabelInit fails. No unmasked node in the graph" << endl;
		return;
	}

	// Initially, all the nodes have label 0
	// The graph contains at east one node, or else, the code
	// fails.
	const int* nodes = mpGraph->Nodes();
	mpFirstNodeWithCertainLabel[0] = nodes[0];
	mpPrevNodeWithCertainLabel[nodes[0]] = -1;
	for (int i = 0; i < unmasked_node_cnt - 1; i++)
	{
		mpFirstNodeWithCertainLabel[i+1] = -1;
		mpNextNodeWithCertainLabel[nodes[i]] = nodes[i + 1];
	}
	for (int i = 1; i < unmasked_node_cnt; i++)
		mpPrevNodeWithCertainLabel[nodes[i]] = nodes[i - 1];
	mpNextNodeWithCertainLabel[nodes[unmasked_node_cnt - 1]] = -1;
	
	// Remove mSource from list corresponding to label 0
	if (-1 == mpPrevNodeWithCertainLabel[mSource])
		mpFirstNodeWithCertainLabel[0] = mpNextNodeWithCertainLabel[mSource];
	else
		mpNextNodeWithCertainLabel[mpPrevNodeWithCertainLabel[mSource]] = mpNextNodeWithCertainLabel[mSource];
	if (-1 != mpNextNodeWithCertainLabel[mSource])
		mpPrevNodeWithCertainLabel[mpNextNodeWithCertainLabel[mSource]] = mpPrevNodeWithCertainLabel[mSource];

	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::GapRelabelInit  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  GapRelabel
 * Description:  mpFirstNodeWithCertainLabel is	an buffer array with length of number of 
 *               nodes. mpFirstNodeWithCertainLabel[i] will indicate the first node	with 
 *               label i, if it is -1, then there is no node with label i. 
 *               mpNextNodeWithCertainLabel and mpPrevNodeWithCertainLabel act as link
 *               lists. mpNextNodeWithCertainLabel[j] stores the next node whose label 
 *               are the same as node j. During the execution of this algorithm, if there 
 *               is an i such that nodeCnt[i] is 0 and there are some nodes, denoted by 
 *               set T, whose mpLabels are greater than i, then we can safely relabel 
 *               all the nodes in T to n (the number of nodes), by gap relabeling heuristic.
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::GapRelabel ( int node, int old_label )
{
	int next;
	int unmasked_node_cnt = mpGraph->NodeCnt();

	// Note, gap relabeling will be effective only to node whose mpLabel is below unmasked_node_cnt
	if (old_label < unmasked_node_cnt)
	{
		// Remove node from old list
		if (-1 == mpPrevNodeWithCertainLabel[node])
			mpFirstNodeWithCertainLabel[old_label] = mpNextNodeWithCertainLabel[node];
		else
			mpNextNodeWithCertainLabel[mpPrevNodeWithCertainLabel[node]] = mpNextNodeWithCertainLabel[node];
		if (-1 != mpNextNodeWithCertainLabel[node])
			mpPrevNodeWithCertainLabel[mpNextNodeWithCertainLabel[node]] = mpPrevNodeWithCertainLabel[node];
	}

	if (mpLabel[node] < unmasked_node_cnt)
	{
		// Insert node in the new list corresponding to the new mpLabel
		next = mpFirstNodeWithCertainLabel[mpLabel[node]];
		mpPrevNodeWithCertainLabel[node] = -1;
		mpNextNodeWithCertainLabel[node] =  next;
		if (-1 != next)
			mpPrevNodeWithCertainLabel[next] = node;
		mpFirstNodeWithCertainLabel[mpLabel[node]] = node;
	}
	// If there is no node with label "old_label", relabel all the node whose label
	// is greater than "old_label" to n
	if (-1 == mpFirstNodeWithCertainLabel[old_label])
	{
		for (int k = old_label + 1; k < unmasked_node_cnt; k++)
		{
			next = mpFirstNodeWithCertainLabel[k];
			while (-1 != next)
			{
				mpLabel[next] = unmasked_node_cnt;
				next = mpNextNodeWithCertainLabel[next];
			}
			mpFirstNodeWithCertainLabel[k] = -1;
		}
	}

	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::GapRelabel  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  ClearFlow
 * Description:  Clear the flow on the edges not being masked in the graph. This function 
 *               is isolated to give more freedom of the algorithm.
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	void
GraphAlgorithm_MaxFlow<FLOW_TYPE>::ClearFlow (  )
{
	const int* edges = mpGraph->Edges();
	for (int i = 0; i < mpGraph->EdgeCnt(); i++)
	{
		int edge = edges[i];
		mpFlow[edge] = 0;
	}
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow::ClearFlow  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  MaxFlow
 * Description:  The FIFO push/rempLabel maximum flow algorithm with gap rempLabeling 
 *               heuristic. ref. Performance analysis and best implementations of old 
 *               and new algorithms for the Open-Pit Mining Problem (1998) by D.S. 
 *               Hochbaum and A. Chen. 
 *       Param:  
 *      Return:
 *       Note :  This algorithm works well on a network with directed loops. And
 *               it always treat the graph as directed.
 *--------------------------------------------------------------------------------------
 */
	template < typename FLOW_TYPE >
	FLOW_TYPE	
GraphAlgorithm_MaxFlow<FLOW_TYPE>::MaxFlow (  )
{
	if (mpGraph->IsNodeMasked(mSource) || mpGraph->IsNodeMasked(mSink))
	{
		cerr << "ERROR : The source or the sink is masked" << endl;
		return 0;
	}

	int unmasked_node_cnt = mpGraph->NodeCnt();
	int all_edge_cnt = mpGraph->EdgeCnt(false);

	const int* nodes = mpGraph->Nodes();
	for (int i = 0; i < unmasked_node_cnt; i++)
	{
		int node = nodes[i];
		mpExcess[node] = 0;
		mpLabel[node] = 0;
		mpIsExcessed[node] = false;
	}
	mpLabel[mSource] = unmasked_node_cnt;

	// Put io edges of a node together
	for (int i = 0; i < unmasked_node_cnt; i++)
	{
		int node = nodes[i];
		mpIsExcessed[node] = false;
		int cnt = 0;
		const int* edges = mpGraph->OutEdges(node);
		for (int j = 0; j < mpGraph->OutDegree(node); j++)
			mpAdjEdges[node][cnt++] = edges[j];
		// If edges are considered as residual edges
		// add all_edge_cnt to distinguish it from normal edge
		edges = mpGraph->InEdges(node);
		for (int j = 0; j < mpGraph->InDegree(node); j++)
			mpAdjEdges[node][cnt++] = edges[j] + all_edge_cnt;
	}

	GapRelabelInit();

	// Based on the flow out of source and the capacity of corresponding edges
	// calculate the mpExcess flow of every node adjacent to the source. 
	// Those nodes serve as the starting elements of the mpExcessedNode.
	int qhead, qtail;
	qhead = qtail = 0;
	const int* edges = mpGraph->OutEdges(mSource);
	for (int i = 0; i < mpGraph->OutDegree(mSource); i++)
	{
		int edge = edges[i];
		int node = mpGraph->ToNode(edge);
		if ( mpLabel[node] < unmasked_node_cnt)
		{
			mpExcess[node] = mpCapacity[edge] - mpFlow[edge];
			mpExcess[mSource] -= mpExcess[node];
			mpFlow[edge] = mpCapacity[edge];
			if (node != mSink)
			{
				mpExcessedNode[qtail++] = node;
				// Remember which nodes have been added into the mpExcessedNode
				mpIsExcessed[node] = true;
			}
		}
	}
	mpIsExcessed[mSource] = true;
	mpIsExcessed[mSink] = true;

	// Initialize mpNextActiveEdgeIndex with the first out edges of every node
	nodes = mpGraph->Nodes();
	for (int i = 0; i < unmasked_node_cnt; i++)
	{
		int node = nodes[i];
		if (0 == mpGraph->IoDegree(node))
			mpNextActiveEdgeIndex[node] = -1;
		else
			mpNextActiveEdgeIndex[node] = 0;
	}

	qtail = qtail % unmasked_node_cnt;
	// Get the next active node from the head of the node mpExcessedNode until the mpExcessedNode is empty
	while (qhead != qtail)
	{
		int node = mpExcessedNode[qhead];
		qhead = (qhead + 1) % unmasked_node_cnt;

		//cout << "Excessed node " << node << endl;
		// Discharge: Push the flow from the node to it's neighbors until it is infeasible

		while (mpExcess[node] > 0)
		{
			if (-1 == mpNextActiveEdgeIndex[node])
			{
				// ReLabel
				int old_label = mpLabel[node];

				if (!Relabel(node))
					break;

				//cout << "Relabel node " << node << " from " << old_label << " to " << mpLabel[node] << endl;
				if (node == 3774)
					//cout << "Stop" << endl;
				GapRelabel(node, old_label);


				if (0 == mpGraph->IoDegree(node))
					mpNextActiveEdgeIndex[node] = -1;
				else
					mpNextActiveEdgeIndex[node] = 0;
			}else{
				int edge = mpAdjEdges[node][mpNextActiveEdgeIndex[node]];
				bool b_residual = false;
				if (edge >= all_edge_cnt)
				{
					edge -= all_edge_cnt;
					b_residual = true;
				}
				int the_other_end = mpGraph->ToNode(edge);
				if (b_residual)
					the_other_end = mpGraph->FromNode(edge);

				if (mpLabel[node] == mpLabel[the_other_end] + 1)
				{
					FLOW_TYPE resi, diff;

					if (b_residual)
						resi = mpFlow[edge];
					else
						resi = mpCapacity[edge] - mpFlow[edge];

					if (resi > mErrorTolerance)
					{
						// Push
						if (mpExcess[node] > resi)
						{
							diff = resi;
							mpExcess[node] -= diff;
						}
						else
						{
							diff = mpExcess[node];
							mpExcess[node] = 0;
						}

						//cout << "Push from " << node << " to " << the_other_end << " with flow " << diff << endl;

						if (b_residual)
							mpFlow[edge] -= diff;
						else
							mpFlow[edge] += diff;

						mpExcess[the_other_end] += diff;
						if (!mpIsExcessed[the_other_end] && the_other_end != mSource && the_other_end != mSink)
						{
							mpExcessedNode[qtail] = the_other_end;
							qtail = (qtail + 1) % unmasked_node_cnt;
							mpIsExcessed[the_other_end] = true;
						}
					}
				}
				mpNextActiveEdgeIndex[node]++;
				if (mpNextActiveEdgeIndex[node] >= mpGraph->IoDegree(node))
					mpNextActiveEdgeIndex[node] = -1;
			}
		}

		mpIsExcessed[node] = false;
	}

	// Calculate the max flow
	FLOW_TYPE max_flow = 0;
	edges = mpGraph->OutEdges(mSource);
	for (int i = 0; i < mpGraph->OutDegree(mSource); i++)
	{
		int edge = edges[i];
		max_flow += mpFlow[edge];
	}
	return max_flow;
}		/* -----  end of method GraphAlgorithm_MaxFlow::MaxFlow  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  MinCut
 * Description:  After MaxFlow is calculated, call this function to get a mincut
 *       Param:  nodes : A cut is defined by a 2-set partition of the nodes in the graph.
 *                       nodes should be an integer array, which will be used to store 
 *                       one set of the partition.
 *               b_min_source : A bool variable. If it is true, a set, containing the 
 *                       source and with minimum size, will be outputed. If it is false,
 *                       a set containing the sink and with minimum size will be outputed.
 *                       The reason is that in most cases, mincut is not unique.
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template < class FLOW_TYPE >
void GraphAlgorithm_MaxFlow<FLOW_TYPE>::MinCut(int* nodes, bool b_min_source)
{
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow<T>::MinCut  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  DFSResidualFromSource
 * Description:  Starting from the source node, do a DFS on the residual network after 
 *               MaxFlow is called. This method will find a minimum set of nodes that
 *               defines a mincut.
 *       Param:  visited : A bool array indexed by internal node ID. It should be 
 *       				   initialized with false for all the elements. After 
 *       				   DFSResidualFromSource finishes its job, all the tags of the nodes
 *       				   been visited in this array will be set to true.
 *      Return:  None
 *--------------------------------------------------------------------------------------
 */
	template < class FLOW_TYPE >
void GraphAlgorithm_MaxFlow<FLOW_TYPE>::DFSResidualFromSource (bool* visited, int curr_node)
{
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow<T>::DFSResidualFromSource  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphAlgorithm_MaxFlow
 *      Method:  DFSResidualFromSink
 * Description:  Starting from the sink node, do a backward DFS on the residual network after 
 *               MaxFlow is called. This method will find a minimum set of nodes that
 *               defines a mincut.
 *       Param:  visited : A bool array indexed by internal node ID. It should be 
 *       				   initialized with false for all the elements. After 
 *       				   DFSResidualFromSink finishes its job, all the tags of the nodes
 *       				   been visited in this array will be set to true.
 *      Return:  None
 *--------------------------------------------------------------------------------------
 */
	template < class FLOW_TYPE >
void GraphAlgorithm_MaxFlow<FLOW_TYPE>::DFSResidualFromSink (bool* visited, int curr_node)
{
	return ;
}		/* -----  end of method GraphAlgorithm_MaxFlow<T>::DFSResidualFromSink  ----- */

#endif
