/*
 * =====================================================================================
 *
 *       Filename:  InstanceWriter.h
 *
 *    Description:  The header file for writing an instance
 *
 *        Version:  1.0
 *        Created:  04/09/2009 12:20:42 PM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */
#ifndef InstanceWriter_H
#define InstanceWriter_H

#include "InstanceHandler.h"

// For setw()
#include <iomanip>
#include <limits>

/*
 * =====================================================================================
 *        Class:  InstanceWriter
 *  Description:  The class for writing an instance
 * =====================================================================================
 */
class InstanceWriter : public InstanceHandler
{
	public:
		// WRITE_HUI : Output the format needed by the matlab code of HuiJiang
		// WRITE_EXP : Output the expression levels of each isoform.
		enum WriteFormat {WRITE_FULL, WRITE_KNOWN, WRITE_EXP, WRITE_TSSPAS, WRITE_EXON, WRITE_ISO, WRITE_V04};

		/* ====================  LIFECYCLE     ======================================= */
		InstanceWriter (ostream* p_output = NULL) : InstanceHandler(p_output) 
		{
			mFormat = WRITE_FULL;
		};
		/* constructor      */

		~InstanceWriter (){};                            /* destructor       */

		void
		SetConfidencelevel(double confi_level){mConfidenceLevel = confi_level;}

		void
		SetWriteFormat(WriteFormat format)
		{
			mFormat = format;
		}

		virtual
		void
		Initialize()
		{
			bHuiHead = true;
		}

		virtual
		void
		CleanUp()
		{
			// To get accurate information of known isoforms, the instances should be
			// got with parameter -use_provied_exon T and don't remove introns
			if (WRITE_KNOWN == mFormat)
			{
				// Calculate the weighted length sum
				double exp_len_sum = 0;
				int64 total_read_cnt = 0;
				for (unsigned k = 0; k < mAllInstances.size(); ++k)
				{
					Instance& an_instance = mAllInstances[k];
					// Only the first read info is effective.
					ReadInfoBase& read_info = *an_instance.mShortReadGroup.mShortReads[0].mpReadInfo;
					total_read_cnt += an_instance.mShortReadGroup.ReadCnt();
					for (unsigned i = 0; i < an_instance.mIsoforms.size(); ++i)
					{
						int len = 0;
						for (unsigned j = 0; j < an_instance.mSegLen.size(); ++j)
							if (an_instance.mIsoforms[i][j]) len += an_instance.mSegLen[j];
						exp_len_sum += an_instance.mIsoExp[i] * (len - read_info.ReadLen() + 1);
					}
				}

				cout << "Total expression weigthed length is " << exp_len_sum << endl;
				cout << "Total number of mapped reads is " << total_read_cnt << endl;

				for (unsigned i = 0; i < mAllInstances.size(); ++i)
					WriteKnown(mAllInstances[i], exp_len_sum, total_read_cnt);
			}
		}

		/* ====================  OPERATORS     ======================================= */
		virtual
		void
		OnInstance (Instance& an_instance)
		{
			switch (mFormat)
			{
			case WRITE_FULL:
				WriteAll(an_instance);
				break;
			case WRITE_KNOWN:
				mAllInstances.push_back(an_instance);
				break;
			case WRITE_EXP:
				WriteExp(an_instance);
				break;
			case WRITE_TSSPAS:
				WriteTSSPAS(an_instance);
				break;
			case WRITE_EXON:
				WriteExon(an_instance);
				break;
			case WRITE_ISO:
				WriteIso(an_instance);
				break;
			case WRITE_V04:
				WriteV04(an_instance);
				break;
			default:
				WriteAll(an_instance);
			}
			InstanceHandler::OnInstance(an_instance);
		}
	protected:

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteExp
		 *  Description:  This methods writes expression of each isoform.
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteExp(Instance& an_instance)
		{
			for (unsigned i = 0; i < an_instance.mIsoforms.size(); i++)
				(*mpOutput) << an_instance.mInstanceCnt << "\t" << an_instance.mGenes[i].mName << "\t" << an_instance.mIsoExp[i] << endl;
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteAll
		 *  Description:  This methods writes all the information of a read. The format should
		 *                be recoganized by a reader
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteAll(Instance& an_instance)
		{
			int 					&instance_cnt =     an_instance.mInstanceCnt;
			vector<int> 			&seg_lens  =        an_instance.mSegLen;
			vector<vector<bool> > 	&isoforms =         an_instance.mIsoforms;
			vector<double> 			&iso_exp =          an_instance.mIsoExp;
			vector<Exon> 			&exons =            an_instance.mExons;
			vector<Gene> 			&genes =        	an_instance.mGenes;
			int                     &known_cnt  =       an_instance.mKnownCnt;
			vector<ShortRead>       &short_reads =      an_instance.mShortReadGroup.mShortReads;

			(*mpOutput) << "Instance\t" << instance_cnt << endl;

			int set_cnt = seg_lens.size();

			(*mpOutput) << "genes\t" << genes.size() << endl;
			for (int i = 0; i < genes.size(); i++)
				genes[i].Write(mpOutput);

			(*mpOutput) << "segs\t" << set_cnt << endl;
			for (int i = 0; i < set_cnt; i++)
				exons[i].Write(mpOutput);

			(*mpOutput) << "len  ";
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << "\t" << setw(6) << seg_lens[i];
			(*mpOutput) << endl;

			(*mpOutput) << "isoforms\t" << isoforms.size() << "\t" << known_cnt << endl;
			for (int i = 0; i < isoforms.size(); i++)
			{
				if (i < known_cnt)
					(*mpOutput) << "K" << setw(4) << i;
				else
					(*mpOutput) << "U" << setw(4) << i;
				for (int j = 0; j < set_cnt; j++)
					(*mpOutput) << "\t" << setw(6) << isoforms[i][j];
				(*mpOutput) << "\t" << setw(6) << iso_exp[i] << endl;
			}

			// Output short read information
			for (unsigned i = 0; i < short_reads.size(); ++i)
			{
				vector<Pattern>& patterns = short_reads[i].mPatterns;
				vector<double>& pattern_dup = short_reads[i].mPatternDup;
				(*mpOutput) << "Reads\t" << patterns.size() << endl;
				for (unsigned j = 0; j < patterns.size(); ++j)
				{
					vector<vector<int> >& segs = patterns[j].mMappedSegs;

					for (unsigned k = 0; k < segs.size(); ++k)
					{
						for (unsigned l = 0; l < segs[k].size(); ++l)
							(*mpOutput) << segs[k][l] << "\t";
						(*mpOutput) << ";";
					}
					(*mpOutput) << pattern_dup[j] << endl;
				}
			}

			// Output start end exons
			(*mpOutput) << "SESegs\t" << an_instance.mStartExons.size() << endl;
			for (unsigned i = 0; i < an_instance.mStartExons.size(); i++)
			{
				(*mpOutput) << an_instance.mStartExons[i][0];
				for (unsigned j = 1; j < an_instance.mStartExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mStartExons[i][j];
				(*mpOutput) << "\t";
				(*mpOutput) << an_instance.mEndExons[i][0];
				for (unsigned j = 1; j < an_instance.mEndExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mEndExons[i][j];
				(*mpOutput) << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteKnown
		 *  Description:  This methods writes all the information of known isoforms
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteKnown(Instance& an_instance, double exp_len_sum, int64 total_read_cnt)
		{
			vector<int> 			&seg_lens  =        an_instance.mSegLen;

			ReadInfoBase& read_info = *an_instance.mShortReadGroup.mShortReads[0].mpReadInfo;

			double curr_ins_len_sum = 0;
			for (unsigned i = 0; i < an_instance.mIsoforms.size(); i++)
			{
				int len = 0;
				for (unsigned j = 0; j < an_instance.mSegLen.size(); j++)
					if (an_instance.mIsoforms[i][j]) len += an_instance.mSegLen[j];
				curr_ins_len_sum += an_instance.mIsoExp[i] * (len - read_info.ReadLen() + 1);
			}
			double curr_ins_read_cnt = an_instance.mShortReadGroup.ReadCnt();

			for (unsigned i = 0; i < an_instance.mIsoforms.size(); i++)
			{
				int len = 0;
				for (unsigned j = 0; j < an_instance.mSegLen.size(); j++)
					if (an_instance.mIsoforms[i][j]) len += an_instance.mSegLen[j];
				double curr_len_sum = an_instance.mIsoExp[i] * (len - read_info.ReadLen() + 1);

				(*mpOutput) << an_instance.mInstanceCnt << "\t";	

				// Output the number of isoforms in this instance
				(*mpOutput) << an_instance.mGenes.size() << "\t";

				// Output expression levels in the instance
				(*mpOutput) << an_instance.mIsoExp[i] << "\t";

				// Please pay attention to the difference between the following two ways of 
				// calculating expression levels in #reads/base and in RPKM. In way 2, the
				// relative relation among different instances is broken. In this way,
				// the randomization among instances is elimiated and it is better to estimate
				// the performance of a method that can calculate isoform expression levels,
				// because whenever the method correctly calculates the relative expression level of
				// different isoforms in the same instance, this method can get the right RPKM
				// of each isoform.
				
				/*
				// Way 1
				// Output expression levels in #read/base
				(*mpOutput) << total_read_cnt / exp_len_sum * an_instance.mIsoExp[i] << "\t";
				
				// Output expression levels in RPKM
				(*mpOutput) << an_instance.mIsoExp[i] * 1000 / exp_len_sum * 1000000 << "\t";
				  */

				if (0 == curr_ins_len_sum) curr_ins_len_sum = 0.0001; // small enough
			
				// Way 2
				// Output expression levels in #read/base
				(*mpOutput) << curr_ins_read_cnt / curr_ins_len_sum * an_instance.mIsoExp[i] << "\t";
				
				// Output expression levels in RPKM
				//(*mpOutput) << curr_ins_read_cnt * 1000000  / total_read_cnt * 
				//	curr_len_sum / curr_ins_len_sum * 1000 / len << "\t";
				(*mpOutput) << curr_ins_read_cnt * 1000000  / total_read_cnt * 
					an_instance.mIsoExp[i] / curr_ins_len_sum * 1000 << "\t";

				// Calculate the mapping which group segments according to current gene.
				vector<int> mapping;
				mapping.resize(an_instance.mSegLen.size());
				vector<Exon>& segs = an_instance.mExons;
				vector<Exon>& exons = an_instance.mGenes[i].mExons;
				int exon_idx = 0;
				int group_cnt = -1;
				vector<bool> grouped_iso;
				grouped_iso.assign(mapping.size(), false);
				vector<double> grouped_seg_len;
				grouped_seg_len.assign(mapping.size(), 0);

				for (int seg_idx = 0; seg_idx < segs.size(); ++seg_idx)
				{
					if (segs[seg_idx].mStart <= exons[exon_idx].mStart ||
						segs[seg_idx].mStart >= exons[exon_idx].mEnd)
						++group_cnt;
					if (segs[seg_idx].mStart >= exons[exon_idx].mStart &&
						segs[seg_idx].mEnd <= exons[exon_idx].mEnd)
					{
						grouped_iso[group_cnt] = true;
					}
					if (segs[seg_idx].mEnd >= exons[exon_idx].mEnd && exon_idx < exons.size() - 1)
						++exon_idx;
					grouped_seg_len[group_cnt] += seg_lens[seg_idx];
					mapping[seg_idx] = group_cnt;
				}
				++group_cnt;
				grouped_iso.resize(group_cnt);
				grouped_seg_len.resize(group_cnt);
				// Now group_cnt is the number of segments in the grouped isoform

				ShortReadGroup grouped_reads = an_instance.mShortReadGroup;
				grouped_reads.Shrink(mapping);

				/*
				cout << "mapping = ";
				for (unsigned j = 0; j < mapping.size(); ++j)
					cout << mapping[j] << "\t";
				cout << endl;
				cout << "known isoform = ";
				for (unsigned j = 0; j < exons.size(); ++j)
					cout << exons[j].mStart << "," << exons[j].mEnd << "\t";
				cout << endl;
				cout << "segments      = ";
				for (unsigned j = 0; j < segs.size(); ++j)
					cout << segs[j].mStart << "," << segs[j].mEnd << "\t";
				cout << endl;
				cout << "new isoform   = ";
				for (unsigned j = 0; j < grouped_iso.size(); ++j)
					cout << grouped_iso[j] << "\t";
				cout << endl;
				cout << "new seglen    = ";
				for (unsigned j = 0; j < grouped_seg_len.size(); ++j)
					cout << grouped_seg_len[j] << "\t";
				cout << endl;
				cout << endl;
				*/

				// Find the first and last exon of current gene
				int first_exon = -1;
				int last_exon = -1;
				for (unsigned j = 0; j < grouped_iso.size(); ++j)
				{
					if (-1 == first_exon && grouped_iso[j]) first_exon = j;
					if (-1 == last_exon && grouped_iso[grouped_iso.size()-j-1]) last_exon = grouped_iso.size()-j-1;
				}

				double min_exp = numeric_limits<double>::max();
				// The min exp without checking the first and the last exon
				double min_exp_nose = min_exp;
				bool b_junc_supported = true;
				int last = -1;
				for (unsigned j = 0; j < grouped_iso.size(); ++j)
				{
					if (!grouped_iso[j]) continue;

					double exp = grouped_reads.SegExp(grouped_seg_len[j], j);
					if (exp < min_exp) min_exp = exp;
					if (exp < min_exp_nose && j != last_exon && j != first_exon) 
						min_exp_nose = exp;

					// A junction is encountered
					if (-1 != last)
					{
						double exp = grouped_reads.JuncExp(last, j);
						if (exp < min_exp) min_exp = exp;
						if (exp < min_exp_nose) 
							min_exp_nose = exp;
						if (exp <= 0) b_junc_supported = false;
					} 
					last = j;
				}

				/*
				Note that the way of deciding whether all the junctions are supported
				by short reads is different from ResultSummary::IsSupportedByJunc
				slightly. For example on the following instance

					Instance	6501
					genes	2
					uc007uvt.1	chr14	-	102032983	102039662	102032983,102036170,102039504,102039605	102033609,102036323,102039516,102039662
					uc007uvu.1	chr14	-	102039272	102039662	102039272,102039605	102039516,102039662
					segs	5
					chr14	+	102032983	102033609	0	1
					chr14	+	102036170	102036323	0	1
					chr14	+	102039272	102039504	0	0
					chr14	+	102039504	102039516	0	1
					chr14	+	102039605	102039662	0	1
					len  	   626	   153	   232	    12	    57
					isoforms	2	2
					K   0	     1	     1	     0	     1	     1	3.12848
					K   1	     0	     0	     1	     1	     1	0.379318
					Reads	4
					0	;11
					2	;1
					1	3	4	;1
					1	;2
					SESegs	2
					0	4
					2	4

				Here the junction (3,4) of the second isoform is not supported by the third short read,
				because the third short read covers the second segment but not the third segment, which
				means that this short read does not belong to the second isoform. In the above "Shrink" 
				method, this short read will be removed.

				However, in method ResultSummary::IsSupportedByJunc, the junction (3,4) of the second
				isoform is supported by the third short read, because ResultSummary::IsSupportedByJunc
				only check whether there is a short read that covers the 3rd and 4th segments at the
				same time.

				Output whether all the junctions of this gene are supported by short reads.
			    */
				
				if (b_junc_supported) (*mpOutput) << "T\t";
				else (*mpOutput) << "F\t";

				// Output the lowest expression level over all the segments and
				// all the junctions
				(*mpOutput) << min_exp << "\t";

				// Output the lowest expression level over all the segments and
				// all the junctions without checking the start and end exons
				// Note that the separator is a comma here.
				(*mpOutput) << min_exp_nose << ";";

				// Output gene content
				an_instance.mGenes[i].Write(mpOutput);
			}
		}


		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteTSSPAS
		 *  Description:  This methods writes the TSSs and PASs of each instance 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteTSSPAS(Instance& an_instance)
		{
			(*mpOutput) << an_instance.mInstanceCnt << "\t"
			            << an_instance.mExons[0].mChr << "\t"
			            << (an_instance.mExons[0].mStrand ? '+' : '-') << "\t"
				        << an_instance.mExons[0].mStart << "\t"
				        << an_instance.mExons[an_instance.mExons.size()-1].mEnd << "\t"
				        << "RANGE" << endl;
			for (unsigned i = 0; i < an_instance.mStartExons.size(); i++)
			{
				(*mpOutput) << an_instance.mInstanceCnt << "\t"
							<< an_instance.mExons[an_instance.mStartExons[i][0]].mStart;
				for (unsigned j = 1; j < an_instance.mStartExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mExons[an_instance.mStartExons[i][j]].mStart;

				(*mpOutput) << "\t" << an_instance.mExons[an_instance.mEndExons[i][0]].mEnd;
				for (unsigned j = 1; j < an_instance.mEndExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mExons[an_instance.mEndExons[i][j]].mEnd;
				(*mpOutput) << "\t" << "TSSPAS" << endl;
			}
			for (unsigned i = 0; i < an_instance.mExons.size(); i++)
			{
				(*mpOutput) << an_instance.mExons[i].mChr << "\t"
							<< (an_instance.mExons[i].mStrand ? '+' : '-') << "\t"
							<< an_instance.mExons[i].mStart << "\tBOUND" << endl;
				(*mpOutput) << an_instance.mExons[i].mChr << "\t"
							<< (an_instance.mExons[i].mStrand ? '+' : '-') << "\t"
							<< an_instance.mExons[i].mEnd << "\tBOUND" << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteIso
		 *  Description:  This methods writes the TSSs and PASs of each instance 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteIso(Instance& an_instance)
		{
			vector<Exon>& exons = an_instance.mExons;
			for (unsigned i = 0; i < an_instance.mIsoforms.size(); i++)
			{
				(*mpOutput) << an_instance.mGenes[i].mName << "\t" << exons[0].mChr << "\t" << (exons[0].mStrand ? '+' : '-') << "\t";
				
				const vector<bool>& an_iso = an_instance.mIsoforms[i];

				int first_idx = 0;
				for (first_idx = 0; first_idx < an_iso.size(); first_idx++)
					if (an_iso[first_idx])
						break;
				int last_idx = 0;
				for (last_idx = an_iso.size() - 1; last_idx >= 0; last_idx--)
					if (an_iso[last_idx])
						break;

				(*mpOutput) << exons[first_idx].mStart << "\t" << exons[last_idx].mEnd << "\t";

				bool b_first = true;
				for (unsigned j = 0; j < an_iso.size(); j++)
				{
					if (an_iso[j])
					{
						if (b_first) (*mpOutput) << exons[j].mStart;
						else (*mpOutput) << "," << exons[j].mStart;
						b_first = false;
					}
				}
				(*mpOutput) << "\t";

				b_first = true;
				for (unsigned j = 0; j < an_iso.size(); j++)
				{
					if (an_iso[j])
					{
						if (b_first) (*mpOutput) << exons[j].mEnd;
						else (*mpOutput) << "," << exons[j].mEnd;
						b_first = false;
					}
				}
				(*mpOutput) << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteExon
		 *  Description:  This methods writes the TSSs and PASs of each instance 
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteExon(Instance& an_instance)
		{
			for (unsigned i = 0; i < an_instance.mExons.size(); i++)
			{
				(*mpOutput) << an_instance.mExons[i].mChr << "\t"
				            << an_instance.mExons[i].mStrand << "\t"
							<< an_instance.mExons[i].mStart << "\t"
				            << an_instance.mExons[i].mEnd  << endl;
			}
		}

		/*
		 *--------------------------------------------------------------------------------------
		 *        Class:  InstanceWriter
		 *       Method:  WriteV04
		 *  Description:  This methods writes all the information of a read. The format should
		 *                be recoganized by a reader
		 *        Param:
		 *       Return:
		 *--------------------------------------------------------------------------------------
		 */
		void
		WriteV04(Instance& an_instance)
		{
			int 					&instance_cnt =     an_instance.mInstanceCnt;
			vector<int> 			&set_sizes =        an_instance.mSegLen;
			vector<vector<bool> > 	&isoforms =         an_instance.mIsoforms;
			vector<double> 			&iso_exp =          an_instance.mIsoExp;
			vector<Exon> 			&exons =            an_instance.mExons;
			vector<Gene> 			&genes =        	an_instance.mGenes;
			int                     &known_cnt  =       an_instance.mKnownCnt;

			(*mpOutput) << "Instance\t" << instance_cnt << "\t" << set_sizes.size() << "\t" << isoforms.size() << "\t" << known_cnt << endl;
			(*mpOutput) << "read len  :\t" << 25 << endl;
			(*mpOutput) << "read cnt  :\t" << 40000000 << endl;
			(*mpOutput) << "cross str :\t" << 1 << endl;
			(*mpOutput) << "noise lev :\t" << 0 << endl;

			int set_cnt = set_sizes.size();
			int iso_cnt = isoforms.size();
			if (exons.size() == set_sizes.size())
			{
				(*mpOutput) << "genes     :\t" << genes.size() << endl;
				for (int i = 0; i < genes.size(); i++)
					genes[i].Write(mpOutput);

				(*mpOutput) << "exp segs  :" << endl;
				for (int i = 0; i < set_cnt; i++)
				{
					Exon& exon = exons[i];
					(*mpOutput) << exon.mChr << "\t" 
						 << (exon.mStrand ? '+' : '-' ) << "\t" 
						 << exon.mStart << "\t" 
						 << exon.mEnd << "\t" 
						 << set_sizes[i] << "\t" 
						 << set_sizes[i] << "\t" 
						 << set_sizes[i] << "\t" 
						 << exon.mStartType << "\t"
						 << exon.mEndType << endl;
				}
			}

			(*mpOutput) << "fit value :";
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << "\t" << setw(6) << setprecision(3) << 0;
			(*mpOutput) << endl;

			(*mpOutput) << "sample cnt:";
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << "\t" << setw(6) << an_instance.mShortReadGroup.ReadCnt(i);
			(*mpOutput) << endl;

			(*mpOutput) << "set size  :";
			for (int i = 0; i < set_cnt; i++)
				(*mpOutput) << "\t" << setw(6) << set_sizes[i];
			(*mpOutput) << endl;

			for (int i = 0; i < iso_cnt; i++)
			{
				if (i < known_cnt)
					(*mpOutput) << "iso  " << setw(4) << i;
				else
					(*mpOutput) << "N iso" << setw(4) << i;
				for (int j = 0; j < set_cnt; j++)
					(*mpOutput) << "\t" << setw(6) << isoforms[i][j];
				(*mpOutput) << "\t" << setw(6) << iso_exp[i] << endl;
			}

			// Output junc reads
			vector<vector<double> > splice_read_cnt;
			an_instance.mShortReadGroup.JuncCnt(set_sizes.size(), splice_read_cnt);
			for (unsigned i = 0; i < splice_read_cnt.size(); i++)
			{
				(*mpOutput) << "           ";
				for (unsigned j = 0; j < splice_read_cnt[i].size(); j++)
					(*mpOutput) << "\t" << setw(6) << splice_read_cnt[i][j];
				(*mpOutput) << endl;
			}

			// Output PE information. None

			// Output start end exons
			(*mpOutput) << "SE exons:\t" << an_instance.mStartExons.size() << endl;
			for (unsigned i = 0; i < an_instance.mStartExons.size(); i++)
			{
				(*mpOutput) << an_instance.mStartExons[i][0];
				for (unsigned j = 1; j < an_instance.mStartExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mStartExons[i][j];
				(*mpOutput) << "\t";
				(*mpOutput) << an_instance.mEndExons[i][0];
				for (unsigned j = 1; j < an_instance.mEndExons[i].size(); j++)
					(*mpOutput) << "," << an_instance.mEndExons[i][j];
				(*mpOutput) << endl;
			}
		}

	private:
		//--------------------------------------------------------------------------------------
		//       Class:  InstanceWriter
		//      Method:  IsSupportedByJunc
		// Description:  Whether a given gene is supported by junction reads
		//  Parameters:  
		//        Note:  exons should be sorted.
		//--------------------------------------------------------------------------------------
		bool
		IsSupportedByJunc(const vector<Exon>& exons, const Gene& isoform, const vector<vector<double> >& junc_cnt)
		{
			const vector<Exon>& gene_exons = isoform.mExons;

			vector<int> start_position;
			start_position.resize(exons.size());
			for (unsigned i = 0; i < start_position.size(); i++)
				start_position[i] = exons[i].mStart;

			vector<int> end_position;
			end_position.resize(exons.size());
			for (unsigned i = 0; i < end_position.size(); i++)
				end_position[i] = exons[i].mEnd;

			for (unsigned i = 0; i < gene_exons.size() - 1; i++)
			{
				int64 end_pos = gene_exons[i].mEnd;
				int64 start_pos = gene_exons[i+1].mStart;

				int start_idx = UtilityTemp<int>::BinarySearch(start_position, start_pos);
				if (start_position[start_idx] != start_pos)
				{
					cerr << "Warning : the known isoform is inconsistent with the expressed segments in this instance" << endl;
					return false;
				}

				int end_idx = UtilityTemp<int>::BinarySearch(end_position, end_pos);
				if (end_position[end_idx] != end_pos)
				{
					cerr << "Warning : the known isoform is inconsistent with the expressed segments in this instance" << endl;
					return false;
				}

				//cout << end_idx << "," << start_idx << "," << junc_cnt[end_idx][start_idx] << endl;
				if (junc_cnt[end_idx][start_idx] <= 0) return false;
			}

			return true;
		}




	private:
		vector<Instance> mAllInstances;
		double mConfidenceLevel;
		WriteFormat mFormat;
		bool bHuiHead;
}; /* -----  end of class InstanceWriter  ----- */

#endif
