// =====================================================================================
// 
//       Filename:  Distribution.hpp
// 
//    Description:  The definition of class Distribution
// 
//        Version:  1.0
//        Created:  02/06/2010 12:31:00 PM
//       Revision:  none
//       Compiler:  g++
// 
//         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
//        Company:  THU
// 
// =====================================================================================

#ifndef Distribution_H
#define Distribution_H

#include <gsl/gsl_cdf.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <cassert>

#include <iostream>

using namespace std;

class Distribution
{
	public:
		virtual bool IsConstant() = 0;
		// Return the probability of the value in range [low, high)
		virtual double ProbInRange(double low, double high) = 0;
		// P{X <= value}
		virtual double ProbLessThan(double value) = 0;
		virtual double GetValue(gsl_rng* rng = 0) = 0;
		virtual double RangeLow() = 0;
		virtual double RangeHigh() = 0;
		virtual void Write(ostream* p_out) = 0;
		virtual void Read(istream* p_in) = 0;
};

class DistributionConstant : public Distribution
{
	public:
		double mValue;

	public:
		virtual bool IsConstant(){return true;}

		virtual double ProbInRange(double low, double high)
		{
			if (mValue >= low && mValue < high) return 1;
			return 0;
		}
		virtual double ProbLessThan(double value)
		{
			if (mValue < value) return 1;
			return 0;
		}
		virtual double GetValue(gsl_rng* rng = 0){return mValue;}

		virtual double RangeLow(){return mValue;}
		virtual double RangeHigh() {return mValue+1;}

		virtual void Write(ostream* p_out)
		{
			(*p_out) << "0" << endl << mValue<< " ";
		}
		virtual void Read(istream* p_in)
		{
			(*p_in) >> mValue;
		}
};

class DistributionGaussian: public Distribution
{
	private:
		double mMean;
		double mStd;

	public:
		virtual bool IsConstant(){return false;}

		virtual double ProbInRange(double low, double high)
		{
			double prob_in_range = gsl_cdf_gaussian_P((high - mMean) / mStd, 1) -
								   gsl_cdf_gaussian_P((low - mMean) / mStd, 1);
			return prob_in_range;
		}
		virtual double GetValue(gsl_rng* rng = 0)
		{
			// Randomly select a value following N(span_mean, span_std)
			if (rng)
			{
				return (gsl_ran_gaussian(rng, 1) * mStd + mMean);
			}
			else
				return mMean;
		}
		virtual double ProbLessThan(double value)
		{
			return gsl_cdf_gaussian_P((value - mMean) / mStd, 1);
		}

		// Allows three standard deviations. About 99.7%
		virtual double RangeLow(){return mMean - 3 * mStd;}
		virtual double RangeHigh() {return mMean + 3 * mStd + 1;}

		virtual void Write(ostream* p_out)
		{
			(*p_out) << "1" << endl;
			(*p_out) << mMean << " ";
			(*p_out) << mStd << " ";
		}
		virtual void Read(istream* p_in)
		{
			(*p_in) >> mMean;
			(*p_in) >> mStd;
		}
};

class DistributionCustomize: public Distribution
{
	private:

	public:
		virtual bool IsConstant(){return false;}

		virtual double ProbInRange(double low, double high)
		{
			return 0;
		}
		virtual double GetValue(gsl_rng* rng = 0)
		{
			return 0;
		}
		virtual double ProbLessThan(double value)
		{
			return 0;
		}

		// Allows three standard deviations. About 99.7%
		virtual double RangeLow(){return 0;}
		virtual double RangeHigh() {return 0;}

		virtual void Write(ostream* p_out)
		{
			(*p_out) << "2" << endl;
		}
		virtual void Read(istream* p_in)
		{
		}
};

#endif
