/*
 * =====================================================================================
 *
 *       Filename:  GraphEx.hpp
 *
 *    Description:  The definition of class GraphEx
 *
 *        Version:  1.0
 *        Created:  04/21/2008 08:01:03 PM
 *       Revision:  11/23/2008 by feeldead
 *       Compiler:  g++
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */
#ifndef GraphEx_h
#define GraphEx_h

#include "Graph.h"
#include <map>
#include "GraphExReader.h"
#include <iostream>

class GraphReader;
class GraphWriter;

using namespace std;
/*
 * =====================================================================================
 *        Class:  GraphEx
 *  Description:  This class is an extension of class Graph. It provides the 
 *                functionalilties of reading, writing a graph and mapping the external
 *                IDs with the inner IDs of a graph. The external IDs are considered as
 *                ExtNodeIDTypes
 * =====================================================================================
 */
template <typename ExtNodeIDType, typename ExtEdgeIDType = ExtNodeIDType>
class GraphEx : public Graph
{
	public:
		/* ====================  LIFECYCLE     ======================================= */
		GraphEx (bool directed = true);                             /* constructor */
		~GraphEx ();


		/* ====================  OPERATORS     ======================================= */

		/* ====================  OPERATIONS    ======================================= */
		virtual void Destroy ();

		// Overload the EndAddNodeOrEdge in Graph
		virtual void EndAddNodeOrEdge();

		int AddEdgeEx (const ExtNodeIDType& startNode, const ExtNodeIDType& endNode, const ExtEdgeIDType& edgeExID);
		int AddNodeEx (const ExtNodeIDType& node);

		/* Read (Create) a graph from the reader */
		virtual void Read (GraphReader* pReader);


		/* ====================  ACCESS        ======================================= */
		ExtNodeIDType GetNodeExID (int inID);
		ExtEdgeIDType GetEdgeExID (int inID);
		int GetNodeInID (ExtNodeIDType exID);
		int GetEdgeInID (ExtEdgeIDType exID);

		void TranslateNodeIDEx2In(const set<ExtNodeIDType> exNodes, set<int>& inNodes);
		void TranslateNodeIDIn2Ex(const set<int> inNodes, set<ExtNodeIDType>& exNodes);
		void TranslateEdgeIDEx2In(const set<ExtEdgeIDType> exEdges, set<int>& inEdges);
		void TranslateEdgeIDIn2Ex(const set<int> inEdges, set<ExtEdgeIDType>& exEdges);


		/* ====================  INQUIRY       ======================================= */

		/* ====================  DATA MEMBERS  ======================================= */
	protected:

		/* Map the external IDs to the inner ID of nodes */
		map<ExtNodeIDType, int>     mmNodeEx2In;

		/* Map the inner IDs to external IDs of nodes */
		ExtNodeIDType*              mpNodeIn2Ex;

		/* Map the external IDs to the inner ID of edges*/
		map<ExtEdgeIDType, int>     mmEdgeEx2In;

		/* Map the inner IDs to external IDs of edges*/
		vector<ExtEdgeIDType>       mEdgeIn2Ex;


	private:

}; /* -----  end of class  GraphEx  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  GraphEx
 * Description:  constructor
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
GraphEx<ExtNodeIDType, ExtEdgeIDType>::GraphEx(bool directed/* = true*/) : Graph(directed)
{
	mpNodeIn2Ex = 0;
}  /* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::GraphEx<ExtNodeIDType, ExtEdgeIDType>  (constructor)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  GraphEx
 * Description:  destructor
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
GraphEx<ExtNodeIDType, ExtEdgeIDType>::~GraphEx()
{
	Destroy();
}  /* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::~GraphEx<ExtNodeIDType, ExtEdgeIDType>  (destructor)  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  Destroy
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
/*virtual*/void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::Destroy ()
{
	if (mpNodeIn2Ex)
	{
		delete[] mpNodeIn2Ex;
		mpNodeIn2Ex = 0;
	}
	
	Graph::Destroy();
	return ;
}		/* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::Destroy  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  void GraphEx<ExtNodeIDType, ExtEdgeIDType>::AddNodeEx
 * Description:  The reader call this function to add a node
 *       Param:  node    :    The external ID of the start node of the edge
 *      Return:  The internal ID of this new node.
 *--------------------------------------------------------------------------------------
 */

template<typename ExtNodeIDType, typename ExtEdgeIDType>
int
GraphEx<ExtNodeIDType, ExtEdgeIDType>::AddNodeEx (const ExtNodeIDType& node)
{
	int in_id;
	typename map<ExtNodeIDType, int>::const_iterator iter;

	iter = mmNodeEx2In.find(node);
	if (iter == mmNodeEx2In.end())
	{
		in_id = AddNode();
		mmNodeEx2In[node] = in_id;
	}
	else
		in_id = mmNodeEx2In[node];

	return in_id;
}		/* -----  end of method GraphEx<T>::AddNodeEx  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  void GraphEx<ExtNodeIDType, ExtEdgeIDType>::AddEdgeEx
 * Description:  The reader call this function to add an edge
 *       Param:  startNode    :    The external ID of the start node of the edge
 *               endNode      :    The external ID of the end node of the edge
 *               edgeExID     :    The external ID of the edge. Default value is "". This
 *                                 function does not handle the repeatition of external IDs
 *                                 of edges.
 *      Return:  The internal ID of this new edge. This ID starts from 0 and increases by 1
 *               each time. Returns -1 if this edge exists before or it is a self loop and
 *               this graph does not allow self loop.
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
int
GraphEx<ExtNodeIDType, ExtEdgeIDType>::AddEdgeEx (const ExtNodeIDType& startNode, 
		const ExtNodeIDType& endNode, const ExtEdgeIDType& edgeExID)
{
	int start, end;
	typename map<ExtNodeIDType, int>::const_iterator iter;

	iter = mmNodeEx2In.find(startNode);
	if (iter == mmNodeEx2In.end())
	{
		start = AddNode();
		mmNodeEx2In[startNode] = start;
	}
	else
		start = mmNodeEx2In[startNode];

	iter = mmNodeEx2In.find(endNode);
	if (iter == mmNodeEx2In.end())
	{
		end = AddNode();
		mmNodeEx2In[endNode] = end;
	}
	else
		end = mmNodeEx2In[endNode];

	// This edge exists
	// Is Edge will treat whether the graph is directed or not
	if (IsEdge(start, end))
	{
#ifdef DEBUG
		//cerr << "DEBUG : existing edge (" << startNode << "," << endNode << ")" << endl;
#endif
		return -1;
	}

	int ein = Graph::AddEdge(start, end);
	// It is a self loop and this graph does not allow self loop
	if (-1 == ein)
		return -1;

	mmEdgeEx2In[edgeExID] = ein;	
	// Use the property that the edge id increases by one each time
	mEdgeIn2Ex.push_back(edgeExID);

	return ein;
}		/* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::AddEdgeEx ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx
 *      Method:  EndAddNodeOrEdge
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	template<typename ExtNodeIDType, typename ExtEdgeIDType>
/*virtual*/	void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::EndAddNodeOrEdge (  )
{
	Graph::EndAddNodeOrEdge();

	// Build the map of inner IDs to exteral IDs of edges and nodes
	// assert mmNodeEx2In.size() = NodeCnt();
	if (mmNodeEx2In.size() != NodeCnt(false))
	{
		cerr << "DEBUG : Wrong: mmNodeEx2In.size() != NodeCnt()" << endl;
	}

	mpNodeIn2Ex = new ExtNodeIDType[NodeCnt(false)];

	typedef map<ExtNodeIDType, int> map_ExtNodeIDType_int;

	for_each_ele_in_group_const_temp (iter, map_ExtNodeIDType_int, mmNodeEx2In)
	{
		if (iter->second < 0)
			cerr << "DEBUG : Wrong: mmNodeEx2In has node id" << iter->first << "," << iter->second << endl;
		mpNodeIn2Ex[iter->second] = iter->first;
	}

	return ;
}		/* -----  end of method GraphEx::EndAddNodeOrEdge  ----- */
/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  void Read (GraphReader* pReader);
 * Description:  Create a graph from the given reader
 *       Param:  reader   :    A pointer to a GraphEx<ExtNodeIDType, ExtEdgeIDType>Reader.
 *      Return:  None
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::Read (GraphReader* pReader)
{
	Destroy();

	// Here the reader must be an object of class GraphExReader.
	GraphExReader* pExReader = dynamic_cast<GraphExReader*>(pReader);
	if (!pExReader)
	{
		cerr << "Create graph from reader failed" << endl;
		cerr << "An instance of class GraphEx<string> is supposed to read  \
				from a GraphExReader. Explictly convert the pointer \
				to the graph to a pointer to its base class and call Read" << endl;
		return ;
	}
	pExReader->SetGraph(this);
	BeginAddNodeOrEdge();
	pExReader->Read();
	EndAddNodeOrEdge();

	return ;
}		/* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::Read (GraphReader* pReader)  ----- */



/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  GetNodeExID (int inID)
 * Description:  Return the external ID of the node specified by the inner ID
 *       Param:  The inner ID of the node
 *      Return:  The external ID of the node
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
ExtNodeIDType
GraphEx<ExtNodeIDType, ExtEdgeIDType>::GetNodeExID (int inID)
{
	if (inID < 0 || inID >= NodeCnt(false))
	{
		cerr << "Invalid inner ID " << inID << " of a node is provided. "
			 << "The number of nodes of the graph is : " << NodeCnt(false) << endl;
	}
	return mpNodeIn2Ex[inID];
}		/* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::GetNodeExID (int inID)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx<ExtNodeIDType, ExtEdgeIDType>
 *      Method:  GetEdgeExID (int inID)
 * Description:  Return the external ID of the edge specified by the inner ID
 *       Param:  The inner ID of the edge
 *      Return:  The external ID of the edge
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
ExtEdgeIDType
GraphEx<ExtNodeIDType, ExtEdgeIDType>::GetEdgeExID (int inID)
{
	if (inID < 0 || inID >= EdgeCnt(false))
	{
		cerr << "Invalid inner ID " << inID << " of an edge is provided. "
			 << "The number of edges with unique IDs is : " << EdgeCnt(false) << endl;
	}
	return mEdgeIn2Ex[inID];
}		/* -----  end of method GraphEx<ExtNodeIDType, ExtEdgeIDType>::GetEdgeExID (int inID)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphIn
 *      Method:  GetEdgeInID (ExtNodeIDType exID)
 * Description:  Return the inner ID of the edge specified by the external ID
 *       Param:  The external ID of the edge 
 *      Return:  The inner ID of the edge 
 *               -1, if no edge has the give external ID.
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
int
GraphEx<ExtNodeIDType, ExtEdgeIDType>::GetEdgeInID (ExtEdgeIDType exID)
{
	typename map<ExtNodeIDType, int>::const_iterator iter;

	iter = mmEdgeEx2In.find(exID);
	if (iter != mmEdgeEx2In.end())
		return iter->second;
	else
		return -1;
}		/* -----  end of method GraphIn::GetEdgeInID (ExtNodeIDType exID)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphIn
 *      Method:  GetNodeInID (ExtNodeIDType exID)
 * Description:  Return the inner ID of the node specified by the external ID
 *       Param:  The external ID of the node 
 *      Return:  The inner ID of the node;
 *               -1, if no node has the give external ID.
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
int
GraphEx<ExtNodeIDType, ExtEdgeIDType>::GetNodeInID (ExtNodeIDType exID)
{
	typename map<ExtNodeIDType, int>::const_iterator iter;

	iter = mmNodeEx2In.find(exID);
	if (iter != mmNodeEx2In.end())
		return iter->second;
	else
		return -1;
}		/* -----  end of method GraphIn::GetNodeInID (ExtNodeIDType exID)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx
 *      Method:  TranslateNodeIDEx2In
 * Description:  Translate a group of external IDs of nodes to internal IDs
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
	void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::TranslateNodeIDEx2In ( const set<ExtNodeIDType> exNodes, set<int>& inNodes )
{
	for_each_ele_in_group_const_temp(iter, set<ExtNodeIDType>, exNodes)
	{
		int in_id = GetNodeInID(*iter);
		if (-1 != in_id)
			inNodes.insert(in_id);
	}
	return ;
}		/* -----  end of method GraphEx::TranslateNodeIDEx2In  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx
 *      Method:  TranslateNodeIDIn2Ex
 * Description:  Translate a group of internal IDs of nodes to exNodes IDs
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
	void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::TranslateNodeIDIn2Ex ( const set<int> inNodes, set<ExtNodeIDType>& exNodes )
{
	for_each_ele_in_group_const(iter, set<int>, inNodes)
	{
		exNodes.insert(GetNodeExID(*iter));
	}
	return ;
}		/* -----  end of method GraphEx::TranslateNodeIDIn2Ex  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx
 *      Method:  TranslateEdgeIDEx2In
 * Description:  Translate a group of external IDs of edges to internal IDs
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
	void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::TranslateEdgeIDEx2In ( const set<ExtEdgeIDType> exEdges, set<int>& inEdges )
{
	for_each_ele_in_group_const_temp(iter, set<ExtEdgeIDType>, exEdges)
	{
		inEdges.insert(GetEdgeInID(*iter));
	}
	return ;
}		/* -----  end of method GraphEx::TranslateEdgeIDEx2In  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  GraphEx
 *      Method:  TranslateEdgeIDIn2Ex
 * Description:  Translate a group of internal IDs of edges to externla edges IDs
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
template<typename ExtNodeIDType, typename ExtEdgeIDType>
	void
GraphEx<ExtNodeIDType, ExtEdgeIDType>::TranslateEdgeIDIn2Ex ( const set<int> inEdges, set<ExtEdgeIDType>& exEdges )
{
	for_each_ele_in_group_const(iter, set<int>, inEdges)
	{
		exEdges.insert(GetEdgeExID(*iter));
	}
	return ;
}		/* -----  end of method GraphEx::TranslateEdgeIDIn2Ex  ----- */

#endif
