/*
 * =====================================================================================
 *
 *       Filename:  GraphAlgorithm_Basic.hpp
 *
 *    Description:  The definition of class GraphAlgorithm_Basic.
 *
 *        Version:  1.0
 *        Created:  03/26/2008 10:35:39 AM
 *       Revision:  none
 *       Compiler:  g++
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef GraphAlgorithm_Basic_h
#define GraphAlgorithm_Basic_h

#include "Graph.h"
#include <list>
#include <vector>
#include <iostream>
#include <set>
#include <limits>
#include "DataStructures.hpp"

using namespace std;

/*
 * =====================================================================================
 *        Class:  GraphAlgorithm_Basic
 *  Description:  Provides a bounch of basic graph algorithms.
 * =====================================================================================
 */
class GraphAlgorithm_Basic
{
	public:
		/*
		static void DFS(Graph* p_graph, int start_node, bool* b_node_visited, 
						int* visited_nodes = 0, int* node_node_visited_cnt = 0,
					    bool* b_edge_visited = 0, int* visited_edges = 0, int* edge_node_visited_cnt = 0);

		static void DFSReverse(Graph* p_graph, int start_node, bool* b_node_visited, 
						int* visited_nodes = 0, int* node_node_visited_cnt = 0,
					    bool* b_edge_visited = 0, int* visited_edges = 0, int* edge_node_visited_cnt = 0);

		static void Components(Graph* p_graph, list<set<int> >& components);
		static double Density(Graph* p_graph, const set<int>& nodes, double* edge_weights = 0, double* node_weights = 0);
		static int MaxDegree(Graph* p_graph, Graph::direction_tag tag = Graph::direction_io);
		static double MaxClique(Graph* p_graph, list<int>& max_clique, double* node_weights, double* edgeweight);
		static long long PathCnt(Graph* p_graph, vector<long long>& cnt_buf, int source, int sink);

		template <typename LENGTH_TYPE>
		static bool ShortestDistance_Dijkstra (Graph* p_graph, int source, int sink, 
		                                         const LENGTH_TYPE* length, LENGTH_TYPE* distance, int* bt_node);

		template <typename LENGTH_TYPE>
		static bool ShortestDistance_BellFord (Graph* p_graph, int source, int sink, 
		                                         const LENGTH_TYPE* length, LENGTH_TYPE* distance, int* bt_node);
  */


	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  DFS
	 * Description:  Starting from a given node, do DFS. Remember the visited nodes.
	 *       Param:  p_graph        :    On which the DFS is operated
	 *               start_node     :    From which node the DFS is operated
	 *               b_node_visited :    Whether each node is visited
	 *               visited_nodes  :    Stores the nodes having been visited.
	 *               node_node_visited_cnt    :    Number of nodes having been visited
	 *               b_edge_visited :    Whether each edge is visited
	 *               visited_edges  :    Stores the edges having been visited
	 *               edge_node_visited_cnt    :    Number of edges having been visited
	 *      Return:
	 *        Note:  Except b_node_visited, all other pointers can be null
	 *--------------------------------------------------------------------------------------
	 */
	static void
	DFS(Graph* p_graph, int start_node, 
							  bool* b_node_visited, int* visited_nodes = 0, int* node_node_visited_cnt = 0,
							  bool* b_edge_visited = 0, int* visited_edges = 0, int* edge_node_visited_cnt = 0)
	{
		// If start_node is visited, do nothing
		if (b_node_visited[start_node]) return;

		b_node_visited[start_node] = true;
		if (visited_nodes && node_node_visited_cnt) visited_nodes[*node_node_visited_cnt] = start_node;
		if (node_node_visited_cnt) (*node_node_visited_cnt)++;

		const int* adjEdges = p_graph->OutEdges(start_node);
		for (int i = 0; i < p_graph->OutDegree(start_node); i++)
		{
			if (b_edge_visited) b_edge_visited[adjEdges[i]] = true;
			if (visited_edges && edge_node_visited_cnt) visited_edges[*edge_node_visited_cnt] = adjEdges[i];
			if (edge_node_visited_cnt) (*edge_node_visited_cnt)++;

			DFS(p_graph, p_graph->ToNode(adjEdges[i]), b_node_visited, visited_nodes, node_node_visited_cnt,
													   b_edge_visited, visited_edges, edge_node_visited_cnt);
		}

		if (p_graph->IsDirected())
			return ;

		adjEdges = p_graph->InEdges(start_node);
		for (int i = 0; i < p_graph->InDegree(start_node); i++)
		{
			if (b_edge_visited) b_edge_visited[adjEdges[i]] = true;
			if (visited_edges && edge_node_visited_cnt) visited_edges[*edge_node_visited_cnt] = adjEdges[i];
			if (edge_node_visited_cnt) (*edge_node_visited_cnt)++;

			DFS(p_graph, p_graph->FromNode(adjEdges[i]), b_node_visited, visited_nodes, node_node_visited_cnt,
													   b_edge_visited, visited_edges, edge_node_visited_cnt);
		}
		return ;
	}		/* -----  end of method GraphAlgorithm_Basic::DFS  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  DFSReverse
	 * Description:  Starting from a given node, do DFS. Remember the visited nodes. Reversely
	 *       Param:  p_graph        :    On which the DFS is operated
	 *               start_node     :    From which node the DFS is operated
	 *               b_node_visited :    Whether each node is visited
	 *               visited_nodes  :    Stores the nodes having been visited.
	 *               node_node_visited_cnt    :    Number of nodes having been visited
	 *               b_edge_visited :    Whether each edge is visited
	 *               visited_edges  :    Stores the edges having been visited
	 *               edge_node_visited_cnt    :    Number of edges having been visited
	 *      Return:
	 *        Note:  Except b_node_visited, all other pointers can be null
	 *--------------------------------------------------------------------------------------
	 */
	static void
	DFSReverse(Graph* p_graph, int start_node, 
							  bool* b_node_visited, int* visited_nodes = 0, int* node_node_visited_cnt = 0,
							  bool* b_edge_visited = 0, int* visited_edges = 0, int* edge_node_visited_cnt = 0)
	{
		// If start_node is visited, do nothing
		if (b_node_visited[start_node]) return;

		b_node_visited[start_node] = true;
		if (visited_nodes && node_node_visited_cnt) visited_nodes[*node_node_visited_cnt] = start_node;
		if (node_node_visited_cnt) (*node_node_visited_cnt)++;

		const int* adjEdges = p_graph->InEdges(start_node);
		for (int i = 0; i < p_graph->InDegree(start_node); i++)
		{
			if (b_edge_visited) b_edge_visited[adjEdges[i]] = true;
			if (visited_edges && edge_node_visited_cnt) visited_edges[*edge_node_visited_cnt] = adjEdges[i];
			if (edge_node_visited_cnt) (*edge_node_visited_cnt)++;

			DFSReverse(p_graph, p_graph->FromNode(adjEdges[i]), b_node_visited, visited_nodes, node_node_visited_cnt,
													   b_edge_visited, visited_edges, edge_node_visited_cnt);
		}
		if (p_graph->IsDirected())
			return ;

		adjEdges = p_graph->OutEdges(start_node);
		for (int i = 0; i < p_graph->OutDegree(start_node); i++)
		{
			if (b_edge_visited) b_edge_visited[adjEdges[i]] = true;
			if (visited_edges && edge_node_visited_cnt) visited_edges[*edge_node_visited_cnt] = adjEdges[i];
			if (edge_node_visited_cnt) (*edge_node_visited_cnt)++;

			DFSReverse(p_graph, p_graph->ToNode(adjEdges[i]), b_node_visited, visited_nodes, node_node_visited_cnt,
													   b_edge_visited, visited_edges, edge_node_visited_cnt);
		}

		return ;
	}		/* -----  end of method GraphAlgorithm_Basic::DFSReverse  ----- */


	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  Components
	 * Description:  Get the connected components of the graph consists of unmasked nodes
	 *               and edges
	 *       Param:  components[out]  :    The set of components
	 *      Return:  None.
	 *--------------------------------------------------------------------------------------
	 */
	static  void
	Components(Graph* p_graph, list<set<int> >& components)
	{
		int node_cnt = p_graph->NodeCnt(false);
		bool* visited_tag = new bool[node_cnt];
		int* visited_nodes = new int[node_cnt];
		int node_visited_cnt;

		const int* unmasked_nodes = p_graph->Nodes();
		for (int i = 0; i < p_graph->NodeCnt(); i++)
			visited_tag[unmasked_nodes[i]] = false;

		for (int i = 0; i < p_graph->NodeCnt(); i++)
		{
			if (visited_tag[unmasked_nodes[i]]) continue;
			node_visited_cnt = 0;
			DFS(p_graph, unmasked_nodes[i], visited_tag, visited_nodes, &node_visited_cnt, NULL, NULL, NULL);

			set<int> nc;
			for (int i = 0; i < node_visited_cnt; i++)
				nc.insert(visited_nodes[i]);
			components.push_back(nc);
		}

		delete[] visited_tag;
		delete[] visited_nodes;
		return ;
	}		/* -----  end of method GraphAlgorithm_Basic::Components  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  Density
	 * Description:  Calculate the density (w(E) / |V|) * (w(V) / |V|) of a given set of nodes
	 *       Param:  p_graph    :    A graph on which the calculation is based on
	 *               nodes     :    A set of internal IDs of the subgraph in the graph
	 *               edge_weights:   An array indexed by internal edge ID storing the weights of edges
	 *               node_weights:   An array indexed by internal node ID storing the weights of edges
	 *      Return:  The density of the specified subgraph
	 *--------------------------------------------------------------------------------------
	 */
	static  double 
	Density(Graph* p_graph, const set<int>& nodes, double* edge_weights/* = 0*/, double* node_weights/* = 0*/)
	{
		set<int> edges;
		p_graph->InducedEdges(nodes, edges);
		double ew = 0;
		for_each_ele_in_group(Iter, set<int>, edges)
		{
			if (edge_weights)
				ew += edge_weights[*Iter];
			else
				ew += 1;
		}

		double nc = nodes.size();
		double nw = 0;
		for_each_ele_in_group_const(Iter, set<int>, nodes)
		{
			if (node_weights)
				nw += node_weights[*Iter];
			else
				nw += 1;
		}
		return (ew / nc) * (nw / nc);
	}		/* -----  end of method GraphAlgorithm_Basic::Density  ----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  MaxDegree
	 * Description:  Get the max degree of a graph
	 *       Param:  tag    :    Refers to direction_tag 
	 *      Return:  The max degree of a graph
	 *--------------------------------------------------------------------------------------
	 */
	static  int
	MaxDegree(Graph* p_graph, Graph::direction_tag tag/* = Graph::direction_io*/)
	{
		int maxdeg = -1;
		const int* nodes = p_graph->Nodes();
		for (int i = 0; i < p_graph->NodeCnt(); i++)
		{
			int curr = -1;
			if (Graph::direction_io == tag)
				curr = p_graph->IoDegree(nodes[i]);
			else if (Graph::direction_in == tag)
				curr = p_graph->InDegree(nodes[i]);
			else
				curr = p_graph->OutDegree(nodes[i]);
			if (curr > maxdeg)
				maxdeg = curr;
		}
		return maxdeg;
	}		/* -----  end of method GraphAlgorithm_Basic::MaxDegree  ----- */


	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  MaxClique
	 * Description:  Find the maximum weighted clique in the graph.
	 *       Param:  max_clique[out]   :   A list store the internal ID of the max weighted
	 *                      graph found.
	 *               node_weights[in]  :   An array of weights for node indexed by internal
	 *                      ID of nodes. If it is 0, every node will be assumed with weight 1
	 *               edge_weights[in]  :   An array of weights for node indexed by internal
	 *                      ID of edges. If it is 0, every edge will be assumed with weight 1
	 *      Return:  The sum of nodes and weights of the clique.
	 *        Note:  Do not pass a large (dense) graph to this function. It will use a brute
	 *               force method to search the optimal one.
	 *--------------------------------------------------------------------------------------
	 */
	static  double 
	MaxClique(Graph* p_graph, list<int>& max_clique, double* node_weights, double* edgeweight)
	{
		return 0;
	}		/* -----  end of method GraphAlgorithm_Basic::MaxClique  ----- */


	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  ShortestDistance_Dijkstra
	 * Description:  Calculate the shortest distance between two given nodes.
	 *       Param:  source   :  The internal ID of from node
	 *               sink     :  The internal ID of to node
	 *               length      :  The length of each edge. This array is indexed by internal
	 *                              ID of edges in the specified graph.
	 *               distance[out]    :  distance[i] stores the length of the shorted path 
	 *                              from node i to source
	 *               bt_node[out] :  After the calculation is done, bt_node[i] stores
	 *                              the next node in the backtracing shorted path from node i 
	 *                              to source. bt_node[source] = bt_node[source].
	 *      Return:  The distance between source and sink.
	 *        Note:  This function adopt Dijkstra algorithm. So, no negative
	 *               length is allowed.
	 *--------------------------------------------------------------------------------------
	 */
	template <typename LENGTH_TYPE>
	static  bool 
	ShortestDistance_Dijkstra (Graph* p_graph, int source, int sink, 
						 const LENGTH_TYPE* length, LENGTH_TYPE* distance, int* bt_node)
	{
		typedef HeapKeyValue<LENGTH_TYPE,int> VertexType;
		BinaryHeap<VertexType*, EleCompare<VertexType*> > biheap;

		vector<VertexType> vertexes;
		vertexes.resize(p_graph->NodeCnt(false));
	/* 
	 * 	const int* nodes = p_graph->Nodes();
	 * 	for (int i = 0; i < p_graph->NodeCnt(); i++)
	 * 	{
	 * 		int node = nodes[i];
	 * 		int edge_id = p_graph->EdgeID(source, node);
	 * 		if (-1 == edge_id && !p_graph->IsEdgeMasked(edge_id))
	 * 		{
	 * 			bt_node[node] = -1;
	 * 			vertexes[node].mKey = numeric_limits<LENGTH_TYPE>::max();
	 * 		}
	 * 		else
	 * 		{
	 * 			vertexes[node].mKey = length[edge_id];
	 * 			bt_node[node] = -source;
	 * 		}
	 * 	}
	 * 	vertexes[source].mKey = 0;
	 * 	bt_node[source] = source;
	 * 
	 * 	for (int i = 0; i < p_graph->NodeCnt(); i++)
	 * 	{
	 * 		// Find the next nearest node
	 * 		int next_near = -1;
	 * 		LENGTH_TYPE next_len = numeric_limits<LENGTH_TYPE>::max();
	 * 		for (int j = 0; j < p_graph->NodeCnt(); j++)
	 * 		{
	 * 			int node = nodes[j];
	 * 			if (bt_node[node] > 0) continue;
	 * 			if (vertexes[node].mKey < next_len)
	 * 			{
	 * 				next_len = vertexes[node].mKey;
	 * 				next_near = node;
	 * 			}
	 * 		}
	 * 
	 * 		if (-1 == next_near) break;
	 * 
	 * 		// Mark this node as visited
	 * 		bt_node[next_near] = -bt_node[next_near];
	 * 
	 * 		// Adjust the distance
	 * 		for (int j = 0; j < p_graph->NodeCnt(); j++)
	 * 		{
	 * 			int node = nodes[j];
	 * 			if (bt_node[node] > 0) continue;
	 * 			int edge_id = p_graph->EdgeID(next_near, node);
	 * 			if (-1 == edge_id && !p_graph->IsEdgeMasked(edge_id)) continue;
	 * 
	 * 			LENGTH_TYPE new_len = distance[next_near] + length[edge_id];
	 * 			if (distance[node] > new_len)
	 * 			{
	 * 				distance[node] = new_len;
	 * 				bt_node[node] = -next_near;
	 * 			}
	 * 		}
	 * 	}
	 * 
	 * 	if (distance[sink] < numeric_limits<LENGTH_TYPE>::max())
	 * 		return true;
	 * 	return false;
	 * 
	 */
		
		const int* edges = p_graph->Edges();
		for (int i = 0; i < p_graph->EdgeCnt(false); i++)
		{
			int edge = edges[i];
			if (!p_graph->IsEdgeMasked(edge) && length[edge] < 0)
			{
				cerr << "Dijsktra algorithm distance check failed. The edge ("<< p_graph->FromNode(edge) << "," << p_graph->ToNode(edge)
					 << ") has negative length: " << length[edge] << endl;
				return false;
			}
#ifdef DEBUG
			cout << "Dijkstra Algo check : length(" << p_graph->FromNode(edge) << "," << p_graph->ToNode(edge) << ") = " << length[edge] << endl;
#endif
		}

		const int* nodes = p_graph->Nodes();
		for (int i = 0; i < p_graph->NodeCnt(); i++)
		{
			int node = nodes[i];
			int edge = p_graph->EdgeID(source, node);
			if (-1 == edge || p_graph->IsEdgeMasked(edge))
			{
				vertexes[node].mKey = numeric_limits<LENGTH_TYPE>::max();
				bt_node[node] = -1;
			}
			else
			{
				vertexes[node].mKey = length[edge];
				bt_node[node] = source;
			}
			vertexes[node].mValue = node;
			if (node == source) continue;
			biheap.push(&vertexes[node]);
		}
		vertexes[source].mKey = 0;
		bt_node[source] = source;

#ifdef DEBUG
		cout << "DEBUG : Init distance" << endl;
		for (int i = 0; i < vertexes.size(); i++)
		{
			cout << vertexes[i].mValue << "\t" << vertexes[i].mKey << endl;
		}
#endif

		while (!biheap.empty())
		{
			VertexType& nearest_v = *biheap.top();

#ifdef DEBUG
			cout << "DEBUG : " << nearest_v.mValue << "\t" << nearest_v.mKey << endl;
#endif

			biheap.pop();

			const int* edges = p_graph->OutEdges(nearest_v.mValue);
			for (int i = 0; i < p_graph->OutDegree(nearest_v.mValue); i++)
			{
				int edge = edges[i];
				int node = p_graph->ToNode(edge);

				if (nearest_v.mKey >= numeric_limits<LENGTH_TYPE>::max() || 
					length[edge] >= numeric_limits<LENGTH_TYPE>::max() )
					continue;
				LENGTH_TYPE new_len = nearest_v.mKey + length[edge];
				if (vertexes[node].mKey > new_len)
				{
#ifdef DEBUG
					cout << "DEBUG : Update distance of node " << node << " from " << vertexes[node].mKey;
#endif
					vertexes[node].mKey = new_len;
					bt_node[node] = nearest_v.mValue;
					biheap.decrease(vertexes[node].heap_ptr());
#ifdef DEBUG
					cout << " to " << vertexes[node].mKey;
					cout << ", because of edge (" << p_graph->FromNode(edge) << "," << p_graph->ToNode(edge) << ") " << length[edge] << endl;
#endif
				}
			}
		}

		for (int i = 0; i < p_graph->NodeCnt(false); i++)
		{
			int node = nodes[i];
			distance[node] = vertexes[node].mKey;
		}

		if (distance[sink] < numeric_limits<LENGTH_TYPE>::max())
			return true;
		return false;

	}		/* -----  end of method GraphAlgorithm_Basic::ShortestDistance_Dijkstra----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  ShortestDistance_BellFord
	 * Description:  Calculate the shortest distance between two given nodes.
	 *       Param:  source   :  The internal ID of from node
	 *               sink     :  The internal ID of to node
	 *               length      :  The length of each edge. This array is indexed by internal
	 *                              ID of edges in the specified graph.
	 *               distance[out]   :  distance[i] stores the length of the shorted path 
	 *                              from source to node i
	 *               bt_node[out] :  After the calculation is done, bt_node[i] stores
	 *                              the next node in the backtracing shorted path from node i 
	 *                              to source. bt_node[source] = bt_node[source].
	 *      Return:  The distance between source and sink.
	 *        Note:  This function adopt BellFord algorithm. Negative length of edges are allowed.
	 *               But no cycle with negative length is allowed.
	 *--------------------------------------------------------------------------------------
	 */
	template <typename LENGTH_TYPE>
	static  bool 
	ShortestDistance_BellFord(Graph* p_graph, int source, int sink, 
						 const LENGTH_TYPE* length, LENGTH_TYPE* distance, int* bt_node)
	{
		const int* nodes = p_graph->Nodes();
		for (int i = 0; i < p_graph->NodeCnt(); i++)
		{
			int node = nodes[i];
			distance[node] = numeric_limits<LENGTH_TYPE>::max();
		}
		distance[source] = 0;
		bt_node[source] = source;

		while (true)
		{
			bool b_changed = false;
			for (int i = 0; i < p_graph->NodeCnt(); i++)
			{
				int node = nodes[i];
				const int* edges = p_graph->InEdges(node);
				for (int j = 0; j < p_graph->InDegree(node); j++)
				{
					int edge = edges[j];
					int f_node = p_graph->FromNode(edge);
					LENGTH_TYPE new_len = distance[f_node] + length[edge];
					if (distance[f_node] != numeric_limits<LENGTH_TYPE>::max() && distance[node] > new_len)
					{
						b_changed = true;
						distance[node] = new_len;
						bt_node[node] = f_node;
					}
				}
			}

			if (!b_changed) break;
		}
		
		if (distance[sink] < numeric_limits<LENGTH_TYPE>::max())
			return true;
		return false;
	}		/* -----  end of method GraphAlgorithm_Basic::ShortestDistance_BellFord----- */

	/*
	 *--------------------------------------------------------------------------------------
	 *       Class:  GraphAlgorithm_Basic
	 *      Method:  PathCnt
	 * Description:  Calculate the number of directed pathes from a source to a sink in a 
	 *               acyclic directed graph
	 *       Param:  cnt_buf   [i/o]  :  An array indexed by the the internal ID of nodes in 
	 *                   the graph. The space should be allocated and all the elements in
	 *                   the array should be initialized to -1 before calling this method.
	 *               source    [i]    :  The internal id of the source
	 *               sink      [i]    :  The internal id of the sink
	 *      Return:  The number of pathes
	 *        Note:  
	 *--------------------------------------------------------------------------------------
	 */
	static  long long
	PathCnt(Graph* p_graph, vector<long long>& cnt_buf, int source, int sink)
	{
		if (!p_graph->IsDirected())
		{
			cerr << "Error, PathCnt works only on directed graphs" << endl;
			return 0;
		}

		if (source == sink) cnt_buf[source] = 1;
		if (cnt_buf[source] >= 0) return cnt_buf[source];

		int curr_cnt = 0;
		const int* out_edges = p_graph->OutEdges(source);
		for (int i = 0; i < p_graph->OutDegree(source); i++)
		{
			int to_node = p_graph->ToNode(out_edges[i]);
			PathCnt(p_graph, cnt_buf, to_node, sink);
			curr_cnt += cnt_buf[to_node];
		}
		cnt_buf[source] = curr_cnt; 
		return curr_cnt;
	}		/* -----  end of method GraphAlgorithm_Basic::PathCnt  ----- */

}; /* -----  end of class  GraphAlgorithm_Basic  ----- */

#endif
