/*
 * =====================================================================================
 *
 *       Filename:  Graph.cc
 *
 *    Description:  The implementation of class Graph.
 *
 *        Version:  1.0
 *        Created:  03/19/2008 09:15:03 PM
 *       Revision:  11/23/2008 by feeldead
 *       Compiler:  g++
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */
#include <iostream>
#include "Graph.h"
#include "GraphReader.h"
#include "GraphWriter.h"

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Graph
 * Description:  constructor
 *       Param:  nodeCnt    :    The estimated number of nodes
 *               edgeCnt    :    The estimated number of edges
 *--------------------------------------------------------------------------------------
 */
Graph::Graph (bool directed/* = true*/)
{
	InitDataMembers();
	mbDirected = directed;
	mbSelfLoopAllowed = false;
}  /* -----  end of method Graph::Graph  (constructor)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Graph
 * Description:  copy constructor
 *--------------------------------------------------------------------------------------
 */
Graph::Graph ( const Graph &other )
{
}  /* -----  end of method Graph::Graph  (copy constructor)  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  ~Graph
 * Description:  destructor
 *--------------------------------------------------------------------------------------
 */
/*virtual*/
Graph::~Graph ()
{
	Destroy();
}  /* -----  end of method Graph::~Graph  (destructor)  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Destroy
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
/*virtual*/void
Graph::Destroy ()
{
	// mpOutDeg_Unmasked serves as a representative for other pointers
	if (mpOutDeg_Unmasked) 
	{
		int i = 0;
		delete[] mpOutDeg_Unmasked;
		delete[] mpInDeg_Unmasked;
		delete[] mpFromNodePos;
		delete[] mpToNodePos;
		for (int i = 0; i < mNodeCnt; i++)
		{
			delete[] mpOutEdges[i];
			delete[] mpInEdges[i];
		}

		delete[] mpOutEdges;
		delete[] mpInEdges;
		delete[] mpFromNode;
		delete[] mpToNode;
		delete[] mpEdgeMask;
		delete[] mpEdges;
		delete[] mpNodes;
		delete[] mpNodeMask;
		delete[] mpOutDeg;
		delete[] mpInDeg;
		delete[] mpNodesPos;
		delete[] mpEdgePos;

		mExistingEdges.clear();

		InitDataMembers();
	}
	return ;
}		/* -----  end of method Graph::Destroy  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  operator =
 * Description:  assignment operator
 *--------------------------------------------------------------------------------------
 */
const Graph&
Graph::operator = ( const Graph &other )
{
	if ( this != &other ) {
	}
	return *this;
}  /* -----  end of method Graph::operator =  (assignment operator)  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  InitDataMembers
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::InitDataMembers (  )
{
	mNodeCnt_saved = mNodeCnt_Unmasked = mNodeCnt = 0;
	mEdgeCnt_saved = mEdgeCnt_Unmasked = mEdgeCnt = 0;
	mpOutDeg_Unmasked = 0;
	mpInDeg_Unmasked = 0;
	mpFromNodePos = 0;
	mpToNodePos = 0;
	mpOutEdges = 0;
	mpInEdges = 0;
	mpFromNode = 0;
	mpToNode = 0;
	mpEdgeMask = 0;
	mpEdges = 0;
	mpNodes = 0;
	mpNodeMask = 0;
	mpOutDeg = 0;
	mpInDeg = 0;
	mbEnableMask = true;

	return ;
}		/* -----  end of method Graph::InitDataMembers  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  BeginAddNodeOrEdge
 * Description:  Call this function before you plan to create a new graph by 
 *				 adding nodes or edges
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	/*virtual*/	void
Graph::BeginAddNodeOrEdge (  )
{
	mNodeCnt_saved = mNodeCnt;
	mEdgeCnt_saved = mEdgeCnt;
	return ;
}		/* -----  end of method Graph::BeginAddNodeOrEdge  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  AddNode
 * Description:  Add a node to the graph, it will return an allocated node's internal ID
 *				 Call this function after BeginAddNodeOrEdge is called
 *       Param:  
 *      Return:  The internal ID of this new node
 *--------------------------------------------------------------------------------------
 */
	int	
Graph::AddNode (  )
{
	return mNodeCnt++;
}		/* -----  end of method Graph::AddNode  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  AddEdge
 * Description:  Add an edge to the graph, it will return an allocated edge's internal ID
 *				 Call this function after BeginAddNodeOrEdge is called
 *       Param:  fromNode    :    The internal ID of the from node of this edge
 *               toNode      :    The internal ID of the to node of this edge
 *      Return:  The internal ID of this new edge. This ID starts from 0 and increases by 1
 *               each time. If it is a selfloop, -1 is returned.
 *        Note:  No duplicated edge is allowed. If a duplicated is intented to be added,
 *               the internal ID of the existing edge will be returned. 
 *               The behaviour of this function is also influnced by whether this graph
 *               is directed or not.
 *--------------------------------------------------------------------------------------
 */
	int
Graph::AddEdge ( int fromNode, int toNode )
{
	// No selfloop is allowed
	if (!mbSelfLoopAllowed && fromNode == toNode)
		return -1;
	map_64_32::const_iterator find_result = 
		mExistingEdges.find(Utility::combine64(fromNode, toNode));
	if (find_result != mExistingEdges.end())
	{
		cerr << "Edge (" << fromNode << "," << toNode << ") exists in the graph" << endl;
		return find_result->second;
	}

	if (!mbDirected)
	{
		find_result = mExistingEdges.find(Utility::combine64(toNode, fromNode));
		if (find_result != mExistingEdges.end())
		{
			cerr << "Edge (" << toNode << "," << fromNode << ") exists in the graph" << endl;
			return find_result->second;
		}
	}

	mExistingEdges.insert(
			map_64_32::value_type(Utility::combine64(fromNode, toNode), mEdgeCnt));
	return mEdgeCnt++;
}		/* -----  end of method Graph::AddEdge  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  EndAddNodeOrEdge
 * Description:  Call this function after you have finished adding nodes and edges.
 *               The status of the graph between calling AddNode (or Add Edge) and
 *               calling EndAddNodeOrEdge is unpredictable.
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
/*virtual*/	void
Graph::EndAddNodeOrEdge (  )
{
	bool* saved_node_mask = 0;
	bool* saved_edge_mask = 0;

	// The graph is unchanged
	if (mNodeCnt == mNodeCnt_saved && mEdgeCnt == mEdgeCnt_saved)
		return;
	if (mNodeCnt < mNodeCnt_saved || mEdgeCnt < mEdgeCnt_saved)
	{
		cerr << "ERROR: something wrong, mNodeCnt > mNodeCnt_saved or \
			mEdgeCnt > mEdgeCnt_saved" << endl;
		cerr << "mNodeCnt_saved  = " << mNodeCnt_saved << endl;
		cerr << "mNodeCnt  = " << mNodeCnt<< endl;
		cerr << "mEdgeCnt_saved  = " << mEdgeCnt_saved << endl;
		cerr << "mEdgeCnt= " << mEdgeCnt<< endl;
		return;
	}
	
	mNodeCnt_Unmasked = mNodeCnt;
	mEdgeCnt_Unmasked = mEdgeCnt;
	// Add new nodes or edges on an existing graph
	if (mNodeCnt_saved != 0)
	{
		delete[] mpOutDeg_Unmasked;
		delete[] mpInDeg_Unmasked;
		delete[] mpFromNodePos;
		delete[] mpToNodePos;

		for (int i = 0; i < mNodeCnt; i++)
		{
			if (mpOutEdges[i]) delete[] mpOutEdges[i];
			if (mpInEdges[i]) delete[] mpInEdges[i];
		}

		delete[] mpOutEdges;
		delete[] mpInEdges;
		delete[] mpFromNode;
		delete[] mpToNode;
		delete[] mpEdges;
		delete[] mpNodes;
		delete[] mpOutDeg;
		delete[] mpInDeg;
		delete[] mpNodesPos;
		delete[] mpEdgePos;

		saved_node_mask = mpNodeMask;
		saved_edge_mask = mpEdgeMask;
	}

	// Setup the data structures according to mExistingEdges;
	// Edge related
	mpEdges = new int[mEdgeCnt];
	mpEdgePos = new int[mEdgeCnt];
	mpFromNode = new int[mEdgeCnt];
	mpToNode = new int[mEdgeCnt];
	mpEdgeMask = new bool[mEdgeCnt];
	mEdgeCnt = 0;
	for_each_ele_in_group_const(iter, map_64_32, mExistingEdges)
	{
		mpEdges[mEdgeCnt] = mEdgeCnt;
		mpEdgePos[mEdgeCnt] = mEdgeCnt;
		mpEdgeMask[mEdgeCnt] = false;
		mpFromNode[iter->second] = Utility::get_combined64_first(iter->first);
		mpToNode[iter->second] = Utility::get_combined64_second(iter->first);
		mEdgeCnt++;
	}

	// Node related
	mpNodes = new int[mNodeCnt];
	mpNodesPos = new int[mNodeCnt];
	mpNodeMask = new bool[mNodeCnt];
	mpOutDeg = new int[mNodeCnt];
	mpOutDeg_Unmasked = new int[mNodeCnt];
	mpInDeg = new int[mNodeCnt];
	mpInDeg_Unmasked = new int[mNodeCnt];
	mpOutEdges = new int*[mNodeCnt];
	mpInEdges = new int*[mNodeCnt];
	for (int i = 0; i < mNodeCnt; i++)
	{
		mpNodes[i] = i;
		mpNodesPos[i] = i;
		mpNodeMask[i] = false;
		mpOutDeg[i] = 0;
		mpInDeg[i] = 0;
	}
	for (int i = 0; i < mEdgeCnt; i++)
	{
		mpOutDeg[mpFromNode[i]]++;
		mpInDeg[mpToNode[i]]++;
	}
	for (int i = 0; i < mNodeCnt; i++)
	{
		mpOutDeg_Unmasked[i] = mpOutDeg[i];
		mpInDeg_Unmasked[i] = mpInDeg[i];
		mpOutEdges[i] = new int[mpOutDeg[i]];
		mpInEdges[i] = new int[mpInDeg[i]];
		mpOutDeg[i] = 0;
		mpInDeg[i] = 0;
	}

	mpFromNodePos = new int[mEdgeCnt];
	mpToNodePos = new int[mEdgeCnt];
	for (int i = 0; i < mEdgeCnt; i++)
	{
		int from = mpFromNode[i];
		int to = mpToNode[i];
		mpOutEdges[from][mpOutDeg[from]] = i;
		mpInEdges[to][mpInDeg[to]] = i;
		mpFromNodePos[i] = mpOutDeg[from];
		mpToNodePos[i] = mpInDeg[to];
		mpOutDeg[from]++;
		mpInDeg[to]++;
	}
	
	// Restore the masks for the node and edges
	if (saved_node_mask)
	{
		mNodeCnt_Unmasked = mNodeCnt;
		mEdgeCnt_Unmasked = mEdgeCnt;
		for (int i = 0; i < mNodeCnt_saved; i++)
			if (saved_node_mask[i]) 
				MaskNode(i, true, false);
		for (int i = 0; i < mEdgeCnt_saved; i++)
			if (saved_edge_mask[i]) 
				MaskEdge(i, true);
		delete[] saved_node_mask;
		delete[] saved_edge_mask;
	}

#ifdef DEBUG0
	cout << __func__ << " : A graph with " << mNodeCnt << " nodes and " << mEdgeCnt << " edges has been created" << endl;
#endif
	return ;
}		/* -----  end of method Graph::EndAddNodeOrEdge  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskNode
 * Description:  Mask or unmask a node specified by its internal ID. You can decide whether
 *				 to mask all the adjacent edges or this node by specifying maskEdges = true
 *				 (for mask) or maskEdges = false (for mask the node only)
 *       Param:  nodes   :    An array of the internal ID of the node to be masked or unmasked
 *               len     :    The length of the array
 *               mask    :    Mask (true) or unmask (false) the node
 *               maskEdges :  Whether  do the same operation on the adjacent edges of this node
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskNode ( const int* nodes, int len, bool mask/* = true*/, bool maskEdges/* = true*/ )
{
	for (int i = 0; i < len; i++)
		MaskNode(nodes[i], mask, maskEdges);
}		/* -----  end of method Graph::MaskNode  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskNode
 * Description:  Mask or unmask a node specified by its internal ID. You can decide whether
 *				 to mask all the adjacent edges or this node by specifying maskEdges = true
 *				 (for mask) or maskEdges = false (for mask the node only)
 *       Param:  nodes   :    An array of the internal ID of the node to be masked or unmasked
 *               mask    :    Mask (true) or unmask (false) the node
 *               maskEdges :  Whether  do the same operation on the adjacent edges of this node
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskNode ( const vector<int>& nodes, bool mask/* = true*/, bool maskEdges/* = true*/ )
{
	for (int i = 0; i < nodes.size(); i++)
		MaskNode(nodes[i], mask, maskEdges);
}		/* -----  end of method Graph::MaskNode  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskEdge
 * Description:  Mask or unmask a edge specified by its internal ID.  
 *       Param:  edges   :    An array of the internal ID of the edges to be masked or unmasked
 *               len     :    The length of the array
 *               mask    :    Mask (true) or unmask (false) the edge 
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskEdge ( const int* edges, int len, bool mask/* = true*/)
{
	for (int i = 0; i < len; i++)
		MaskEdge(edges[i], mask);
}		/* -----  end of method Graph::MaskEdge  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskEdge
 * Description:  Mask or unmask a edge specified by its internal ID.
 *       Param:  edges   :    An array of the internal ID of the edge to be masked or unmasked
 *               mask    :    Mask (true) or unmask (false) the edge
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskEdge ( const vector<int>& edges, bool mask/* = true*/)
{
	for (int i = 0; i < edges.size(); i++)
		MaskEdge(edges[i], mask);
}		/* -----  end of method Graph::MaskEdge  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskNode
 * Description:  Mask or unmask a node specified by its internal ID. You can decide whether
 *				 to mask all the adjacent edges or this node by specifying maskEdges = true
 *				 (for mask) or maskEdges = false (for mask the node only)
 *       Param:  node    :    The internal ID of the node to be masked or unmasked
 *               mask    :    Mask (true) or unmask (false) the node
 *               maskEdges :  Whether do the same operation on the adjacent edges of this node
 *
 *        Note:  CRITICAL : Do NOT call this function when you are looping over all
 *               the nodes in the graph by
 *               'const int* nodes = Graph::Nodes()' or
 *               'const int* edges = Graph::Edges()'
 *               Because the contents in 'nodes' or 'edges' will be changed by 
 *               Graph::MaskNode.
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskNode ( int node, bool mask/* = true*/, bool maskEdges/* = true*/ )
{
	if (node < 0 || node >= mNodeCnt)
	{
		cerr << "The ID of node is " << node << ", which is out of range" << endl;
		return;
	}
	if (mpNodeMask[node] == mask)
		return;
	mpNodeMask[node] = mask;

	// Switch the node array, such that all the unmasked nodes are before
	// the masked nodes
	int pos = mpNodesPos[node];
	if (mask)
	{
		mNodeCnt_Unmasked--;
		UtilityTemp<int>::Switch(mpNodes[pos], mpNodes[mNodeCnt_Unmasked]);
		UtilityTemp<int>::Switch(mpNodesPos[node], mpNodesPos[mpNodes[pos]]);
	}else
	{
		UtilityTemp<int>::Switch(mpNodes[pos], mpNodes[mNodeCnt_Unmasked]);
		UtilityTemp<int>::Switch(mpNodesPos[node], mpNodesPos[mpNodes[pos]]);
		mNodeCnt_Unmasked++;
	}

	if (maskEdges)
	{
		if (mask)
		{
			// Out edges
			int umasked_node_cnt = mpOutDeg_Unmasked[node];
			mpOutDeg_Unmasked[node] = 0;
			for (int i = 0; i < umasked_node_cnt; i++)
			{
				int edge = mpOutEdges[node][i];
				if (mpEdgeMask[edge]) continue;
				mpEdgeMask[edge] = true;
				int pos = mpEdgePos[edge];
				if (mask)
				{
					mEdgeCnt_Unmasked--;
					UtilityTemp<int>::Switch(mpEdges[pos], mpEdges[mEdgeCnt_Unmasked]);
					UtilityTemp<int>::Switch(mpEdgePos[edge], mpEdgePos[mpEdges[pos]]);
					MaskEdge_InEdgeList(edge, true);
				}
			}

			// In edges
			umasked_node_cnt = mpInDeg_Unmasked[node];
			mpInDeg_Unmasked[node] = 0;
			for (int i = 0; i < umasked_node_cnt; i++)
			{
				int edge = mpInEdges[node][i];
				if (mpEdgeMask[edge]) continue;
				mpEdgeMask[edge] = true;
				int pos = mpEdgePos[edge];
				if (mask)
				{
					mEdgeCnt_Unmasked--;
					UtilityTemp<int>::Switch(mpEdges[pos], mpEdges[mEdgeCnt_Unmasked]);
					UtilityTemp<int>::Switch(mpEdgePos[edge], mpEdgePos[mpEdges[pos]]);
					MaskEdge_OutEdgeList(edge, true);
				}
			}
		}
		else // Unmask
		{
			vector<int> adj_edges;
			adj_edges.reserve(mpOutDeg[node] + mpInDeg[node]);
			for (int i = 0; i < mpOutDeg[node]; i++)
				adj_edges.push_back(mpOutEdges[node][i]);
			for (int i = 0; i < mpInDeg[node]; i++)
				adj_edges.push_back(mpInEdges[node][i]);
			for_each_ele_in_group_const(iter, vector<int>, adj_edges)
				MaskEdge(*iter, false);
		}
	}
	
	//cout << "DEBUG : " << __func__ << " end" << endl;
	return ;
}		/* -----  end of method Graph::MaskNode  ----- */
		

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskEdge_OutEdgeList
 * Description:  Mask or unmask an edge in the out edge list of its from node
 *       Param:  edge    :    The internal ID of the edge to be masked or unmasked
 *               mask    :    Mask (true) or unmask (false) the edge
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskEdge_OutEdgeList ( int edge, bool mask )
{
	int node = mpFromNode[edge];
	int* edges = mpOutEdges[node];
	int pos = mpFromNodePos[edge];
	if (mask)
	{
		mpOutDeg_Unmasked[node]--;
		UtilityTemp<int>::Switch(edges[pos], edges[mpOutDeg_Unmasked[node]]);
		UtilityTemp<int>::Switch(mpFromNodePos[edge], mpFromNodePos[edges[pos]]);
	}
	else
	{
		UtilityTemp<int>::Switch(edges[pos], edges[mpOutDeg_Unmasked[node]]);
		UtilityTemp<int>::Switch(mpFromNodePos[edge], mpFromNodePos[edges[pos]]);
		mpOutDeg_Unmasked[node]++;
	}
	return ;
}		/* -----  end of method Graph::MaskEdge_OutEdgeList  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskEdge_InEdgeList
 * Description:  Mask or unmask an edge in the in edge list of its to node
 *       Param:  edge    :    The internal ID of the edge to be masked or unmasked
 *               mask    :    Mask (true) or unmask (false) the edge
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskEdge_InEdgeList ( int edge, bool mask )
{
	int node = mpToNode[edge];
	int* edges = mpInEdges[node];
	int	pos = mpToNodePos[edge];
	if (mask)
	{
		mpInDeg_Unmasked[node]--;
		UtilityTemp<int>::Switch(edges[pos], edges[mpInDeg_Unmasked[node]]);
		UtilityTemp<int>::Switch(mpToNodePos[edge], mpToNodePos[edges[pos]]);
	}else
	{
		UtilityTemp<int>::Switch(edges[pos], edges[mpInDeg_Unmasked[node]]);
		UtilityTemp<int>::Switch(mpToNodePos[edge], mpToNodePos[edges[pos]]);
		mpInDeg_Unmasked[node]++;
	}
	return ;
}		/* -----  end of method Graph::MaskEdge_InEdgeList  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskEdge
 * Description:  Mask or unmask an edge specifying by its internal ID. If either one 
 *               of the end nodes of this edge is masked the unmasking operation will
 *               unmask the end nodes
 *
 *       Param:  edge    :    The internal ID of the edge to be masked or unmasked
 *               mask    :    Mask (true) or unmask (false) the edge
 *
 *        Note:  CRITICAL : Do NOT call this function when you are looping over all
 *               the nodes in the graph by
 *               'const int* nodes = Graph::Nodes()' or
 *               'const int* edges = Graph::Edges()'
 *               Because the contents in 'nodes' or 'edges' will be changed by 
 *               Graph::MaskEdge.
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskEdge ( int edge, bool mask/* = true*/)
{
	if (edge < 0 || edge >= mEdgeCnt)
	{
		cerr << "The id of edge is " << edge << ", which is out of range" << endl;
		return;
	}
	if (mpEdgeMask[edge] == mask)
		return;

	int pos = mpEdgePos[edge];
	if (mask)
	{
		mpEdgeMask[edge] = true;
		mEdgeCnt_Unmasked--;
		UtilityTemp<int>::Switch(mpEdges[pos], mpEdges[mEdgeCnt_Unmasked]);
		UtilityTemp<int>::Switch(mpEdgePos[edge], mpEdgePos[mpEdges[pos]]);
		MaskEdge_OutEdgeList(edge, true);
		MaskEdge_InEdgeList(edge, true);
	}else if (!mpNodeMask[mpFromNode[edge]] && !mpNodeMask[mpToNode[edge]])
	{
		MaskNode(mpFromNode[edge], false, false);
		MaskNode(mpToNode[edge], false, false);
		mpEdgeMask[edge] = false;
		UtilityTemp<int>::Switch(mpEdges[pos], mpEdges[mEdgeCnt_Unmasked]);
		UtilityTemp<int>::Switch(mpEdgePos[edge], mpEdgePos[mpEdges[pos]]);
		mEdgeCnt_Unmasked++;
		MaskEdge_OutEdgeList(edge, false);
		MaskEdge_InEdgeList(edge, false);
	}

	return ;
}		/* -----  end of method Graph::MaskEdge  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskAll
 * Description:  Mask or unmask all the nodes and edges. This function is faster than calling
 *				 MaskNode or MaskEdge multiple times
 *       Param:  mask    :    Mask (true) or unmask (false) the whe whole graph.
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskAll ( bool mask )
{
	if (mask)
	{
		for (int i = 0; i < mNodeCnt_Unmasked; i++)
			mpNodeMask[mpNodes[i]] = true;
		mNodeCnt_Unmasked = 0;
		for (int i = 0; i < mEdgeCnt_Unmasked; i++)
			mpEdgeMask[mpEdges[i]] = true;
		mEdgeCnt_Unmasked = 0;
		for (int i = 0; i < mNodeCnt; i++)
		{
			mpInDeg_Unmasked[i] = 0;
			mpOutDeg_Unmasked[i] = 0;
		}
	}
	else
	{
		for (int i = mNodeCnt_Unmasked; i < mNodeCnt; i++)
			mpNodeMask[mpNodes[i]] = false;
		mNodeCnt_Unmasked = mNodeCnt;
		for (int i = mEdgeCnt_Unmasked; i < mEdgeCnt; i++)
			mpEdgeMask[mpEdges[i]] = false;
		mEdgeCnt_Unmasked = mEdgeCnt;
		for (int i = 0; i < mNodeCnt; i++)
		{
			mpInDeg_Unmasked[i] = mpInDeg[i];
			mpOutDeg_Unmasked[i] = mpOutDeg[i];
		}
	}
	return ;
}		/* -----  end of method Graph::MaskAll  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  MaskAllEdges
 * Description:  Mask or unmask all the edges. This function is faster than calling
 *				 MaskEdge multiple times
 *       Param:  mask    :    Mask (true) or unmask (false) all the edges.
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::MaskAllEdges ( bool mask )
{
	if (mask)
	{
		for (int i = 0; i < mEdgeCnt_Unmasked; i++)
			mpEdgeMask[mpEdges[i]] = true;
		mEdgeCnt_Unmasked = 0;
		for (int i = 0; i < mNodeCnt; i++)
		{
			mpInDeg_Unmasked[i] = 0;
			mpOutDeg_Unmasked[i] = 0;
		}
	}
	else
	{
		for (int i = mEdgeCnt_Unmasked; i < mEdgeCnt; i++)
			mpEdgeMask[mpEdges[i]] = false;
		mEdgeCnt_Unmasked = mEdgeCnt;
		for (int i = 0; i < mNodeCnt; i++)
		{
			mpInDeg_Unmasked[i] = mpInDeg[i];
			mpOutDeg_Unmasked[i] = mpOutDeg[i];
		}
	}
	return ;
}		/* -----  end of method Graph::MaskAll  ----- */




/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Write
 * Description:  Write the graph to the writer 
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::Write ( GraphWriter* pWriter )
{
	pWriter->SetGraph(this);
	pWriter->Write();
	return ;
}		/* -----  end of method Graph::Write  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  AdjEdges
 * Description:  Get the adjacent edges of a specified node.
 *       Param:  node    :    The internal ID of the node.
 *               edges   :    A list storing the internal IDs of edges
 *               tag     :    Refers to direction_tag
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::AdjEdges ( int node, list<int>& edges, direction_tag tag/* = direction_io*/ )
{
	if (direction_out == tag || direction_io == tag)
		for (int i = 0; i < mpOutDeg_Unmasked[node]; i++)
			edges.push_back(mpOutEdges[node][i]);
	if (direction_in == tag || direction_io == tag)
		for (int i = 0; i < mpInDeg_Unmasked[node]; i++)
			edges.push_back(mpInEdges[node][i]);
	return ;
}		/* -----  end of method Graph::AdjEdges  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  AdjEdges
 * Description:  Get the adjacent edges of a specified node.
 *       Param:  node    :    The internal ID of the node.
 *               edges   :    A vector storing the internal IDs of edges
 *               tag     :    Refers to direction_tag
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::AdjEdges ( int node, vector<int>& edges, direction_tag tag/* = direction_io*/ )
{
	if (direction_out == tag || direction_io == tag)
		for (int i = 0; i < mpOutDeg_Unmasked[node]; i++)
			edges.push_back(mpOutEdges[node][i]);
	if (direction_in == tag || direction_io == tag)
		for (int i = 0; i < mpInDeg_Unmasked[node]; i++)
			edges.push_back(mpInEdges[node][i]);
	return ;
}		/* -----  end of method Graph::AdjEdges  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Neighbors
 * Description:  Get the neighbors of the specified node
 *       Param:  node    :    The internal ID of the node.
 *               neigh   :    A list storing the internal IDs of neighbors of "node"
 *               tag     :    Refers to direction_tag
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::Neighbors ( int node, set<int>& neigh, direction_tag tag/* = direction_io*/ )
{
	if (direction_out == tag || direction_io == tag)
		for (int i = 0; i < OutDegree(node); i++)
			neigh.insert(mpToNode[mpOutEdges[node][i]]);
	if (direction_in == tag || direction_io == tag)
		for (int i = 0; i < InDegree(node); i++)
			neigh.insert(mpFromNode[mpInEdges[node][i]]);
	return ;
}		/* -----  end of method Graph::AdjEdges  ----- */



/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Neighbors
 * Description:  Get the neighbors of the specified node
 *       Param:  node    :    The internal ID of the node.
 *               neigh   :    A list storing the internal IDs of neighbors of "node"
 *               tag     :    Refers to direction_tag
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::Neighbors ( int node, list<int>& neigh, direction_tag tag/* = direction_io*/ )
{
	if (direction_out == tag || direction_io == tag)
		for (int i = 0; i < OutDegree(node); i++)
			neigh.push_back(mpToNode[mpOutEdges[node][i]]);
	if (direction_in == tag || direction_io == tag)
		for (int i = 0; i < InDegree(node); i++)
			neigh.push_back(mpFromNode[mpInEdges[node][i]]);
	return ;
}		/* -----  end of method Graph::AdjEdges  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Neighbors
 * Description:  Get the neighbors of the specified node
 *       Param:  node    :    The internal ID of the node.
 *               neigh   :    A vector storing the internal IDs of neighbors of "node"
 *               tag     :    Refers to direction_tag
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::Neighbors ( int node, vector<int>& neigh, direction_tag tag/* = direction_io*/ )
{
	if (direction_out == tag || direction_io == tag)
		for (int i = 0; i < OutDegree(node); i++)
			neigh.push_back(mpToNode[mpOutEdges[node][i]]);
	if (direction_in == tag || direction_io == tag)
		for (int i = 0; i < InDegree(node); i++)
			neigh.push_back(mpFromNode[mpInEdges[node][i]]);
	return ;
}		/* -----  end of method Graph::AdjEdges  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Neighbors
 * Description:  Get the neighbors of a set of nodes.
 *       Param:  nodes    :    The set of nodes specified
 *               neigh    :    The neighbors of "nodes". Note, neigh may contains nodes
 *                             in "nodes".
 *               tag      :    Refers to direction_tag
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::Neighbors ( const set<int>& nodes, set<int>& neigh, direction_tag tag/* = direction_io*/ )
{
	for_each_ele_in_group_const(iter, set<int>, nodes)
	{
		if (direction_out == tag || direction_io == tag)
			for (int i = 0; i < OutDegree(*iter); i++)
				neigh.insert(mpToNode[mpOutEdges[*iter][i]]);
		if (direction_in == tag || direction_io == tag)
			for (int i = 0; i < InDegree(*iter); i++)
				neigh.insert(mpFromNode[mpOutEdges[*iter][i]]);
	}

	return ;
}		/* -----  end of method Graph::Neighbors  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  InducedEdges
 * Description:  
 *       Param:  nodes    :    The set of nodes specified
 *               edges    :    All the edges whose both ends are in the "nodes"
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::InducedEdges ( const set<int>& nodes, set<int>& edges )
{
	for_each_ele_in_group_const(iter, set<int>, nodes)
	{
		for (int i = 0; i < mpOutDeg_Unmasked[*iter]; i++)
			if (nodes.find(mpToNode[mpOutEdges[*iter][i]]) != nodes.end())
				edges.insert(mpOutEdges[*iter][i]);
		for (int i = 0; i < mpInDeg_Unmasked[*iter]; i++)
			if (nodes.find(mpFromNode[mpInEdges[*iter][i]]) != nodes.end())
				edges.insert(mpInEdges[*iter][i]);
	}
	return ;
}		/* -----  end of method Graph::InducedEdges  ----- */


/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  EdgeID
 * Description:  Get the internal ID of an edge specified by its start node and end node.
				 This function is influenced by whether the graph is directed or not
 *       Param:  fromNode    :    The internal ID of the from node of this edge
 *               toNode      :    The internal ID of the to node of this edge
 *      Return:  The internal ID of the edge. -1 if this edge doesn't exist.
 *--------------------------------------------------------------------------------------
 */
	int	
Graph::EdgeID ( int fromNode, int toNode )
{
	map_64_32::const_iterator find_result = 
		mExistingEdges.find(Utility::combine64(fromNode, toNode));
	if (find_result != mExistingEdges.end())
		return find_result->second;
	if (!mbDirected)
	{
		find_result = mExistingEdges.find(Utility::combine64(toNode, fromNode));
		if (find_result != mExistingEdges.end())
			return find_result->second;
	}
	return -1;
}		/* -----  end of method Graph::EdgeID  ----- */

/*
 *--------------------------------------------------------------------------------------
 *       Class:  Graph
 *      Method:  Check
 * Description:  
 *       Param:  
 *      Return:
 *--------------------------------------------------------------------------------------
 */
	void
Graph::Check (  )
{
	for (int i = 0; i < mNodeCnt_Unmasked; i++)
		if (mpNodeMask[mpNodes[i]])
			cerr << "ERROR : node " << i << " is masked wrong" << endl;
	for (int i = mNodeCnt_Unmasked; i < mNodeCnt; i++)
		if (!mpNodeMask[mpNodes[i]])
			cerr << "ERROR : node " << i << " is unmasked wrong" << endl;
	for (int i = 0; i < mEdgeCnt_Unmasked; i++)
		if (mpEdgeMask[mpEdges[i]])
			cerr << "ERROR : edge " << i << " is masked wrong" << endl;
	for (int i = mEdgeCnt_Unmasked; i < mEdgeCnt; i++)
		if (!mpEdgeMask[mpEdges[i]])
			cerr << "ERROR : edge " << i << " is unmasked wrong" << endl;
	for (int i = 0; i < mNodeCnt; i++)
	{
		int* edges = mpOutEdges[i];
		for (int j = 0; j < mpOutDeg_Unmasked[i]; j++)
			if (mpEdgeMask[edges[j]])
				cerr << "ERROR : out edge " << edges[j] << " of node " << i << " is masked wrong" << endl;
		for (int j = mpOutDeg_Unmasked[i]; j < mpOutDeg[i]; j++)
			if (!mpEdgeMask[edges[j]])
				cerr << "ERROR : out edge " << edges[j] << " of node " << i << " is unmasked wrong" << endl;
		edges = mpInEdges[i];
		for (int j = 0; j < mpInDeg_Unmasked[i]; j++)
			if (mpEdgeMask[edges[j]])
				cerr << "ERROR : in edge " << edges[j] << " of node " << i << " is masked wrong" << endl;
		for (int j = mpInDeg_Unmasked[i]; j < mpInDeg[i]; j++)
			if (!mpEdgeMask[edges[j]])
				cerr << "ERROR : in edge " << edges[j] << " of node " << i << " is unmasked wrong" << endl;
	}
	return ;
}		/* -----  end of method Graph::Check  ----- */

