/*
 * =====================================================================================
 *
 *       Filename:  Algorithms.hpp
 *
 *    Description:  Implement some common algorithms
 *
 *        Version:  1.0
 *        Created:  01/13/2009 03:56:17 PM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Jianxing Feng (feeldead), feeldead@gmail.com
 *        Company:  THU
 *
 * =====================================================================================
 */

#ifndef Algorithms_h
#define Algorithms_h

#include <set>
#include <vector>
#include <map>

#include "Utility.hpp"
#include "DataStructures.hpp"

using namespace std;

namespace Algorithms
{
	/* 
	 * ===  FUNCTION  ======================================================================
	 *         Name:  SetCover
	 *  Description:  Given an instance of set cover, this function resolve the set cover
	 *                problem with a greedy strategy
	 *        Param:  elements  :  The universe of the elements
	 *                sets      :  The given sets
	 *                set_weigths : The weights of each set
	 *                selected_sets  :  As output. Which sets are selected in the cover
	 *       Return:
	 * =====================================================================================
	 */
	template <typename Type>
	static void SetCover(vector<Type>& elements, vector<set<Type> >& sets, 
			vector<double>& set_weigths, vector<int>& selected_sets)
	{

#ifdef DEBUG2
		cout << __func__ << " : elements cnt = " << elements.size() << " sets cnt = " << sets.size() << endl;
#endif
		typedef HeapKeyValue<double,int> Key2Value;
		BinaryHeap<Key2Value*, EleCompare<Key2Value*> > biheap;

		vector<int> set_sizes;
		vector<Key2Value> key2values;
		key2values.reserve(sets.size());
		set_sizes.reserve(sets.size());
		selected_sets.reserve(sets.size());
		map<Type, vector<int> > ele2sets;

		for (int i = 0; i < sets.size(); i++)
		{
			set_sizes[i] = sets[i].size();
			if (sets[i].size() == 0) continue;

			// Initialize the heap
			key2values[i].mKey = (double)set_weigths[i]/sets[i].size();
			key2values[i].mValue = i;
			biheap.push(&key2values[i]);

			// Remember which sets contain a given element
			for_each_ele_in_group_temp(iter, set<Type>, sets[i])
			{
				if (ele2sets.find(*iter) == ele2sets.end())
				{
					vector<int> a_set;
					a_set.push_back(i);
					ele2sets[*iter] = a_set;
				}
				else
				{
					ele2sets[*iter].push_back(i);
				}
			}
		}


		// Remember which elements been covered
		map<Type,bool> covered_eles;
		for (int i = 0; i < elements.size(); i++)
			covered_eles[elements[i]] = false;

		int covered_cnt = 0;
		while (!biheap.empty())
		{
			// Select current set
			Key2Value& curr = *biheap.top();
			selected_sets.push_back(curr.mValue);

			for_each_ele_in_group_temp(iter, set<Type>, sets[curr.mValue])
			{
				if (!covered_eles[*iter])
				{
					covered_cnt++;
					covered_eles[*iter] = true;

					// Update the heap
					vector<int>& rel_sets = ele2sets[*iter];
					for (int i = 0; i < rel_sets.size(); i++)
					{
						int set_idx = rel_sets[i];
						key2values[set_idx].mKey = 0;				
						biheap.decrease(key2values[set_idx].heap_ptr());
						biheap.pop();

						set_sizes[set_idx]--;
						if (set_sizes[set_idx] != 0)
						{
							key2values[set_idx].mKey = (double)set_weigths[set_idx]/set_sizes[set_idx];				
							biheap.push(&key2values[set_idx]);
						}
					}
				}
			}

#ifdef DEBUG2
			cout << __func__ << ": " << curr.mValue << " es " << elements.size() << " cn " << covered_cnt << endl;
#endif
			if (elements.size() == covered_cnt) break;
		}

		if (elements.size() != covered_cnt)
			cerr << __func__ << ": Not all the elements have been covered" << endl;
	}
};

#endif
